<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use App\Models\Point;
use App\Models\Society;
use App\Models\Sharing;
use App\Models\Itinerary;
use Illuminate\Support\Facades\DB;


class AdminItineraryController extends Controller
{


    public function index()
    {
        $itineraries = \App\Models\Itinerary::with([
            'society:id,name',
            'assignedUsers:id,firstname,lastname,email'
            
        ])
        ->orderBy('created_at','desc')
        ->paginate(20);
    
        return view('admin.itins.index', compact('itineraries'));
    }



    /**
     * Display the itinerary creation form.
     */

    /** Show the admin create form */
    public function create(Request $request)
    {
 
        $societies = Society::query()
        ->whereHas('users', fn($q) => $q->where('role_id', 2))
        ->orderBy('name')
        ->get(['id','name']);

        return view('admin.itins.create', compact('societies'));
    }





    /**
     * Store a new itinerary.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'               => ['required', 'string', 'max:255', 'unique:itineraries,name'],
            'assigned_user_id'   => ['required', 'exists:users,id'],
            'society_id'         => ['required', 'exists:societydb,id'],
            'assigned_user_ids'  => ['nullable', 'array'],
            'assigned_user_ids.*'=> ['integer', 'exists:users,id'],

            // 👇 Liste ordonnée des waypoints
            'point_ids'          => ['nullable','array'],
            'point_ids.*'        => ['integer','exists:points,id'],
        ]);

        $itinerary = \App\Models\Itinerary::create([
            'name'             => $data['name'],
            'owner_user_id'    => $request->user()->id,
            'assigned_user_id' => $data['assigned_user_id'],
            'society_id'       => $data['society_id'],
            'scheduled_date'   => now(),
        ]);

        // Assigned users (pivot itinerary_user)
        $assignedUserIdsInput = $request->input('assigned_user_ids', []);
        if (!is_array($assignedUserIdsInput)) {
            $assignedUserIdsInput = array_filter([$assignedUserIdsInput]);
        }

        $assignedUserIds = collect($assignedUserIdsInput)
            ->filter(fn($v) => is_numeric($v))
            ->map(fn($v) => (int) $v)
            ->filter(fn($v) => $v > 0)
            ->unique()
            ->values();
        $itinerary->assignedUsers()->sync($assignedUserIds->all());

        if ($request->has('point_ids')) {
            // 1) Nettoyage côté serveur
            $pointIds = collect($request->input('point_ids', []))
                ->filter(fn($v) => is_numeric($v))     // garde numériques
                ->map(fn($v) => (int) $v)              // cast int
                ->filter(fn($v) => $v > 0)             // supprime 0 / négatifs
                ->unique()
                ->values();

            $validIds = Point::whereIn('id', $pointIds)->pluck('id');

            // 3) Journaliser si on a écarté des IDs invalides (ex: 0)
            $invalid = $pointIds->diff($validIds);
            if ($invalid->isNotEmpty()) {
                Log::warning('Itinerary points: filtered invalid IDs', [
                    'itinerary_id' => $itinerary->id ?? null,
                    'invalid'      => $invalid->values(),
                ]);
            }
        
            // 4) Construire les données pivot (sort_order = position)
            $syncData = [];
            foreach ($validIds->values() as $idx => $pid) {
                $syncData[$pid] = ['sort_order' => $idx + 1];
            }

            $itinerary->points()->sync($syncData);
        }

        // $ids = array_unique($data['assigned_user_ids'] ?? []);
        // if ($ids) $it->users()->sync($ids);

        // // waypoints + sort_order
        // $pointIds = array_values(array_unique($data['point_ids'] ?? []));
        // if (!empty($pointIds)) {
        //     $attach = [];
        //     foreach ($pointIds as $i => $pid) {
        //         $attach[$pid] = ['sort_order' => $i + 1];
        //     }
        //     $it->points()->syncWithoutDetaching($attach);
        // }

        return redirect()->route('admin.itins.index', ['locale'=>app()->getLocale()])
            ->with('status', __('Itinerary successfully created.'));
    }




    /**
     * AJAX: list company users (id > 1) for a given society_id
     * Route name: admin.itins.usersForSociety
     */
    public function usersForSociety(Request $request)
    {
        $data = $request->validate([
            'society_id' => ['required','integer','exists:'.(new Society)->getTable().',id'],
        ]);
        $societyId = (int) $data['society_id'];

        // Users belonging to that society (via pivot societydb_user), excluding id=1
        $users = User::where('id','>',1)
            ->whereHas('societies', fn($q) => $q->where('societydb.id', $societyId)) 
            ->orderBy('lastname')->orderBy('firstname')
            ->get(['id','firstname','lastname','email']);

        return response()->json($users);
    }




    /** Return Pro Admins (role_id = 2) linked to a society */
    public function adminsForSociety(Request $request)
    {

        \Log::info('adminsForSociety: hit', [
            'auth_id' => auth()->id(),
            'cookie_sid' => request()->cookie(session()->getName()), // laravel_session
            'session_id' => session()->getId(),
            'path' => request()->path(),
            'query' => request()->query(),
            'expects_json' => request()->expectsJson(),
          ]);


        $data = $request->validate([
            'society_id' => ['required','integer','exists:societydb,id'],
        ]);
    
        $societyId = (int) $data['society_id'];
    
        // Pro Admins (role_id = 2) liés à cette société
        $admins = User::query()
            ->select('users.id','users.firstname','users.lastname','users.email')
            ->join('societydb_user','societydb_user.user_id','=','users.id')
            ->where('societydb_user.society_id', $societyId)
            ->where('users.role_id', 2) // pro admin
            ->orderBy('users.lastname')
            ->orderBy('users.firstname')
            ->get();
        return response()->json($admins);
    }




    public function edit(Request $request, \App\Models\Itinerary $itinerary)
    {
        // Load relations for view
        // $itinerary->load(['society:id,name', 'assignedUsers:id,firstname,lastname,email', 'points:id,name']);
    
        // // Admin Pros (role_id = 2)
        // $adminPros = \App\Models\User::where('role_id', 2)
        //     ->with(['societies:id,name'])
        //     ->orderBy('lastname')->orderBy('firstname')
        //     ->get(['id','firstname','lastname','email']);
    
        // // Target society = current itinerary->society_id
        // $targetSocietyId = (int) $itinerary->society_id ?: null;
    
        // // Company users (id > 1) for assigned checkboxes
        // $companyUsers = collect();
        // if ($targetSocietyId) {
        //     $companyUsers = \App\Models\User::where('id','>',1)
        //         ->whereHas('societies', fn($q) => $q->where('societydb.id', $targetSocietyId))
        //         ->orderBy('lastname')->orderBy('firstname')
        //         ->get(['id','firstname','lastname','email']);
        // }
    
        // // For quick checkbox pre-check
        // $assignedIds = $itinerary->assignedUsers->pluck('id')->all();
    
        // return view('admin.itins.edit', compact('itinerary','adminPros','companyUsers','targetSocietyId','assignedIds'));

        $itinerary->load([
            'assignedUsers:id,firstname,lastname,email',
            'points:id,name', // via belongsToMany
        ]);
    
        // Pour alimenter le select Sociétés (id + name)
        $societies = Society::orderBy('name')->get(['id','name']);
    
        // Pro admin pré-sélectionné : on part du premier assignedUser avec role_id=2 si existant,
        // sinon null. (Tu peux adapter selon ta logique.)
        $preAssignedAdminId = optional(
            $itinerary->assignedUsers->firstWhere('role_id', 2)
        )->id;
    
        // Points déjà sélectionnés (ordre pivot sort_order)
        $selectedPoints = $itinerary->points()
            ->select('points.id','points.name')
            ->withPivot('sort_order')
            ->orderBy('itinerary_points.sort_order')
            ->get()
            ->map(fn($p) => ['id'=>$p->id, 'name'=>$p->name]);
    
        return view('admin.itins.edit', [
            'itinerary'          => $itinerary,
            'societies'          => $societies,
            'preAssignedAdminId' => $preAssignedAdminId,
            'selectedPoints'     => $selectedPoints,
        ]);
    }
    




    public function update(Request $request, Itinerary $itinerary)
    {
        $data = $request->validate([
            'name'              => ['required','string','max:255','unique:itineraries,name,'.$itinerary->id],
            'society_id'        => ['required','integer','exists:societydb,id'],
            'assigned_user_id'  => ['nullable','integer','exists:users,id'], // si tu veux conserver une colonne "assigned_user_id"
            'assigned_user_ids' => ['nullable','array'],
            'assigned_user_ids.*' => ['integer','exists:users,id'],
            'point_ids'         => ['array'],        // optionnel côté création si tu enregistres ici
            'point_ids.*'       => ['integer','exists:points,id'],
        ]);

        // MAJ champs simples
        $itinerary->fill([
            'name'       => $data['name'],
            'society_id' => $data['society_id'],
        ]);

        if (array_key_exists('assigned_user_id', $data)) {
            $itinerary->assigned_user_id = $data['assigned_user_id'] ?: null;
        }

        $itinerary->save();

        // Re-synchronisation des utilisateurs assignés :
        // si tu veux que "assigned_user_id" soit un ancien vestige ET que la vraie
        // assignation passe par la table pivot itineraries_users :
        $assignedUserIdsInput = $request->input('assigned_user_ids', []);
        if (!is_array($assignedUserIdsInput)) {
            $assignedUserIdsInput = array_filter([$assignedUserIdsInput]);
        }

        $ids = collect($assignedUserIdsInput)
            ->filter(fn($v) => is_numeric($v))
            ->map(fn($v) => (int) $v)
            ->filter(fn($v) => $v > 0)
            ->unique()
            ->values()
            ->all();
        $itinerary->assignedUsers()->sync($ids);

        // Points + ordre
        if ($request->has('point_ids')) {
            
            $pointIds = collect($request->input('point_ids', []))
                ->filter(fn($v) => is_numeric($v))     // garde numériques
                ->map(fn($v) => (int) $v)              // cast int
                ->filter(fn($v) => $v > 0)             // supprime 0 / négatifs
                ->unique()
                ->values();
        

            $validIds = Point::whereIn('id', $pointIds)->pluck('id');

            // 3) Journaliser si on a écarté des IDs invalides (ex: 0)
            $invalid = $pointIds->diff($validIds);
            if ($invalid->isNotEmpty()) {
                Log::warning('Itinerary points: filtered invalid IDs', [
                    'itinerary_id' => $itinerary->id ?? null,
                    'invalid'      => $invalid->values(),
                ]);
            }
        
            // 4) Construire les données pivot (sort_order = position)
            $syncData = [];
            foreach ($validIds->values() as $idx => $pid) {
                $syncData[$pid] = ['sort_order' => $idx + 1];
            }
            $itinerary->points()->sync($syncData);
        }

        return redirect()
            ->route('admin.itins.edit', ['itinerary'=>$itinerary->id])
            ->with('ok', __('Itinerary updated.'));
    }
    




    public function destroy(Request $request, \App\Models\Itinerary $itinerary)
    {
        // Detach pivot first (good hygiene)
        $itinerary->assignedUsers()->detach();
    
        // If you have itinerary_points etc., clean them too:
        // \DB::table('itinerary_points')->where('itinerary_id',$itinerary->id)->delete();
    
        $itinerary->delete();
    
        return redirect()
            ->route('admin.itins.index', ['locale' => app()->getLocale()])
            ->with('ok', __('Itinerary deleted.'));
    }




    public function map(\Illuminate\Http\Request $request, \App\Models\Itinerary $itinerary)
    {
        $itinerary->load([
            'society:id,name',
            'points:id,name,latitude,longitude', 
        ]);

        return view('admin.itins.map', compact('itinerary'));
    }



    /**
     * Return waypoints for the selected society + those shared to the chosen pro admin.
     * - Points “propriétaires” : tous les users de la société (owner dans society)
     * - Points “partagés” : Sharing.target_user_id = admin_id (l’admin pro choisi)
     */
    public function pointsForSociety(Request $request)
    {
        try {
            $sid = (int) $request->query('society_id');
            $aid = (int) $request->query('admin_id');

            if ($sid <= 0 || $aid <= 0) {
                return response()->json(['message' => 'Missing parameters'], 422);
            }

            // Members of the society
            $memberIds = \DB::table('societydb_user')
                ->where('society_id', $sid)
                ->pluck('user_id');

            // Points partagés (table sharingdb)
            $sharedPointIds = Sharing::query()
                ->where('target_user_id', $aid)
                ->pluck('point_id');

            $points = Point::query()
                ->select('points.id','points.name')
                ->where(function ($query) use ($memberIds, $sharedPointIds) {
                    $hasCondition = false;

                    if ($memberIds->isNotEmpty()) {
                        $query->whereIn('points.user_id', $memberIds);
                        $hasCondition = true;
                    }

                    if ($sharedPointIds->isNotEmpty()) {
                        $method = $hasCondition ? 'orWhereIn' : 'whereIn';
                        $query->{$method}('points.id', $sharedPointIds);
                        $hasCondition = true;
                    }

                    if (!$hasCondition) {
                        // Aucun point éligible
                        $query->whereRaw('1 = 0');
                    }
                })
                ->distinct()
                ->orderBy('points.name')
                ->get();

            // Minimal JSON to the front
            return response()->json(
                $points->map(fn($p) => [
                    'id'   => (int) $p->id,
                    'name' => (string) ($p->name ?? ('#'.$p->id)),
                ])->values()
            );
        } catch (\Throwable $e) {
            Log::error('pointsForSociety: exception', [
                'sid' => $request->query('society_id'),
                'aid' => $request->query('admin_id'),
                'err' => $e->getMessage(),
            ]);
            return response()->json([
                'message' => 'Server error',
            ], 500);
        }
    }




    // JSON: waypoints visibles pour le manager choisi
    public function pointsForManager(Request $request)
    {
        $data = $request->validate([
            'manager_id' => ['required','integer','exists:users,id'],
        ]);
        $manager = User::findOrFail($data['manager_id']);

        Log::info('pointsForManager: incoming', [
            'manager_id' => $manager->id,
            'auth_id'    => optional($request->user())->id,
        ]);

        try {
            // If you have a scope: visibleFor($user). If not, fallback to owned points only.
            $q = Point::query()
                // ->visibleFor($manager)
                ->where('user_id', $manager->id)
                ->orderBy('name');

            $points = $q->get(['id','name','latitude','longitude']);

            Log::info('pointsForManager: result', [
                'count' => $points->count()
            ]);

            return response()->json($points);
        } catch (\Throwable $e) {
            Log::error('pointsForManager: exception', ['err' => $e->getMessage()]);
            return response()->json(['error' => 'server_error', 'message' => $e->getMessage()], 500);
        }
    }


}
