<?php
namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Point;
use App\Models\User;
use App\Services\PointQr;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\PointController; 
use Illuminate\Validation\Rule;

class AdminPointController extends Controller
{


    // List waypoints (useful for /admin/points)
    public function index()
    {
        $points = Point::latest()->paginate(50);
        return view('admin.points.index', compact('points'));
    }


    // Point detail page (used by admin.points.show)
    public function show(Point $point)
    {
        return view('admin.points.show', compact('point'));
    }

    
    // Edit a waypoint (used when the views link here)
    public function edit(string $locale, Point $point)
    {
        $users = User::query()
            ->whereIn('role_id', [1, 2])             // 👈 admin et admins pro
            ->with(['societies:id,name'])             // charge les sociétés pour les pro
            ->orderByRaw("FIELD(role_id, 1, 2)")      // admin en premier
            ->orderBy('lastname')
            ->orderBy('firstname')
            ->get(['id','role_id','firstname','lastname','email']);

        return view('admin.points.edit', compact('point','users'));
    }
        




    // Update (stub so calling the route does not trigger a 500 error)
    public function update(string $locale, Point $point)
    {
        // Autorisation si tu as une policy (optionnel mais conseillé)
        // $this->authorize('update', $point);

        $data = request()->validate([
            // … tes autres champs …
            'user_id' => ['required', 'integer', Rule::exists('users','id')],
        ]);

        // si 'user_id' n’est pas dans $fillable, utilises forceFill()
        $point->forceFill([
            // … autres affects …
            'user_id' => $data['user_id'],
        ])->save();

        return redirect()
            ->route('admin.points.edit', ['locale'=>$locale, 'point'=>$point->id])
            ->with('ok', __('Creator updated.'));
    }


    

    // One-shot action: regenerate missing (or empty) QR codes
    public function generateMissingQr(Request $request)
    {
        @set_time_limit(0);
        @ini_set('memory_limit', '512M');

        $updated = 0; $errors = 0;

        Point::whereNull('qrcode')->orWhere('qrcode', '')
            ->orderBy('id')
            ->chunkById(500, function ($chunk) use (&$updated, &$errors) {
                foreach ($chunk as $p) {
                    try {
                        PointQr::generateFor($p, true);
                        $updated++;
                    } catch (\Throwable $e) {
                        $errors++;
                        Log::error('QR generation failed (admin bulk)', [
                            'point_id' => $p->id,
                            'err' => $e->getMessage(),
                        ]);
                    }
                }
            });

        $msg = [];
        if ($updated) $msg[] = "$updated QR généré(s)";
        if ($errors)  $msg[] = "$errors échec(s) (voir logs)";
        if (!$msg)    $msg[] = "Aucun QR à générer";

        return back()->with('status', implode(' · ', $msg).'.');
    }






    // Bulk variant with a "max" value coming from the form (optional)
    public function bulkGenerateQrcodes(Request $request)
    {
        $max = max(1, (int)($request->input('max', 200)));
        @set_time_limit(0);
        @ini_set('memory_limit', '512M');

        $updated = 0; $errors = 0;

        Point::whereNull('qrcode')->orWhere('qrcode', '')
            ->orderBy('id')
            ->limit($max)
            ->chunk(200, function ($points) use (&$updated, &$errors) {
                foreach ($points as $p) {
                    try {
                        PointQr::generateFor($p, true);
                        $updated++;
                    } catch (\Throwable $e) {
                        $errors++;
                        Log::error('QR generation failed (admin bulk limit)', [
                            'point_id' => $p->id,
                            'err' => $e->getMessage(),
                        ]);
                    }
                }
            });

        $msg = [];
        if ($updated) $msg[] = "$updated QR généré(s)";
        if ($errors)  $msg[] = "$errors échec(s) (voir logs)";
        if (!$msg)    $msg[] = "Aucun QR à générer";

        return back()->with('status', implode(' · ', $msg).'.');
    }





    public function generateSerials(Request $request)
{
    $updated = 0;

    \App\Models\Point::whereNull('serial')->orWhere('serial','')
        ->orderBy('id')
        ->chunkById(500, function ($chunk) use (&$updated) {
            foreach ($chunk as $point) {
                $point->serial = PointController::makeSerial($point); // 👈
                $point->save();
                $updated++;
            }
        });

    return back()->with('status', $updated
        ? "{$updated} serial(s) généré(s)."
        : "Aucun serial à générer : tout est déjà rempli.");
}


}
