<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class AdminUserController extends Controller
{
    
    public function index(\Illuminate\Http\Request $request)
    {
        $q            = trim((string) $request->query('q', ''));
        $societyId    = (int) $request->query('society_id', 0) ?: null;

        // Societies with users count (for filter buttons)
        $societies = Society::withCount(['users' => function ($q) {
            // you can add extra constraints if needed
        }])->orderBy('name')->get(['id','name']);

        $users = User::with('societies:id,name')
            ->where('id', '>', 1)
            ->when($q !== '', function ($qb) use ($q) {
                $qb->where(function ($w) use ($q) {
                    $w->where('firstname','like',"%{$q}%")
                    ->orWhere('lastname','like',"%{$q}%")
                    ->orWhere('email','like',"%{$q}%");
                });
            })
            ->when($societyId, function ($qb) use ($societyId) {
                // filter by pivot society
                $qb->whereHas('societies', fn($w) => $w->where('societydb.id', $societyId));
            })
            ->orderBy('lastname')->orderBy('firstname')
            ->paginate(30)
            ->appends([
                'q' => $q,
                'society_id' => $societyId,
            ]);

        return view('admin.users.index', [
            'users'        => $users,
            'societies'    => $societies,
            'q'            => $q,
            'societyId'    => $societyId,   // current filter
        ]);
    }



    // Shallow stub to avoid 404/500 when the views link here
    public function show(User $user)
    {
        return view('admin.users.show', compact('user'));
    }






    public function edit(?string $locale = null, User $user)
    {
        $roles = Role::orderBy('name')->get(['id','name']);
        $societies = Society::orderBy('name')->get(['id','name']);

        // ⚑ Toujours défini. old() prioritaire si retour validation.
        $selectedSocietyIds = old(
            'society_ids',
            $user->societies()
                 ->pluck('societydb.id')   // nom de table explicite pour éviter l’ambiguïté
                 ->all()
        );

        return view('admin.users.edit', compact('user','roles','societies','selectedSocietyIds'));
    }






    public function update(Request $request, ?string $locale = null, User $user)
    {
        $data = $request->validate([
            'firstname'        => ['nullable','string','max:255'],
            'lastname'         => ['nullable','string','max:255'],
            'email'            => ['required','email','max:255','unique:users,email,'.$user->id],
            'address1'         => ['nullable','string','max:255'],
            'address2'         => ['nullable','string','max:255'],
            'postcode'         => ['nullable','string','max:255'],
            'city'             => ['nullable','string','max:255'],
            'country_isocode'  => ['nullable','string','max:3'],
            'password'         => ['nullable','string','min:8'],
            'photo'            => ['nullable','image','max:2048'],
            'role_id'          => ['nullable','exists:roledb,id'],
            'societies'        => ['array'],
            'societies.*'      => ['nullable','integer','exists:societydb,id'],
        ]);

        $name = trim($data['firstname'] ?? $user->name);

        // Validate your other fields (except 'blocked')
        // $data = $request->except(['set_blocked']); // don't mass-assign 'blocked'
        $data = $request->except(['password', 'password_confirmation', 'set_blocked']);

        $user->fill($data);

        // Checkbox is intent: checked → now(), unchecked → null
        $shouldBlock = $request->boolean('set_blocked');
        $wasBlocked  = !is_null($user->blocked);
        $user->blocked = $shouldBlock ? now() : null;

        $user->fill([
            'firstname' => $data['firstname'] ?? $user->firstname,
            'lastname'  => $data['lastname']  ?? $user->lastname,
            'email'     => $data['email']     ?? $user->email,
            'name'      => $name ,
            'address1'  => $data['address1']  ?? $user->address1,
            'address2'  => $data['address2']  ?? $user->address2,
            'postcode'  => $data['postcode']  ?? $user->postcode,
            'city'      => $data['city']      ?? $user->city,
            'country_isocode' => $data['country_isocode'] ?? $user->country_isocode,
            'blocked'   => $user->blocked,
        ]);

        // Photo
        if ($request->hasFile('photo')) {
            if ($user->photo && Storage::disk('public')->exists($user->photo)) {
                Storage::disk('public')->delete($user->photo);
            }

            $user->photo = $request->file('photo')->store('avatars', 'public');
        }

        // Password
        if (!empty($data['password'])) {
            $data['password'] = bcrypt($data['password']);
        } else {
            unset($data['password']);
        }
        // if ($request->filled('password')) {
        //     $request->validate(['password' => ['string','min:8','confirmed']]);
        //     $user->password = Hash::make($request->input('password'));
        // }

        $user->role_id = $data['role_id'] ?? null;
        $user->save();
    
        $rawSocietyIds = $data['societies'] ?? [];

        $ids = collect($rawSocietyIds)
            ->filter(fn($v) => filled($v))
            ->map(fn($v) => (int)$v)
            ->unique()
            ->values()
            ->all();

        $user->societies()->sync($ids);

        // kill sessions if just blocked
        if ($shouldBlock && ! $wasBlocked) {
            if (config('session.driver') === 'database') {
                DB::table('sessions')->where('user_id', $user->id)->delete();
            }
            if (Auth::id() === $user->id) {
                Auth::logout();
                $request->session()->invalidate();
                $request->session()->regenerateToken();
                return redirect()->route('login', ['locale' => app()->getLocale()])->withErrors([
                    'email' => __('Your account has been blocked. Contact support.'),
                ]);
            }
        }

        return back()->with('ok', $shouldBlock
            ? __('User has been blocked.')
            : __('User has been unblocked.')
        );

        // return redirect()
        //     ->route('admin.users.edit', $user)   // ou ->route('admin.users.index')
        //     ->with('ok', __('User updated.'));
    }







    /** Show CSV import form (Admin only) */
    public function importForm()
    {
        return view('admin.users.import');
    }




    /**
     * Import users from CSV (Admin only).
     *
     * Supported headers (case-insensitive):
     *   - required: email
     *   - optional: firstname, lastname, password, role_id, blocked, society_id, society_name
     * Behavior:
     *   - Upsert by email (LOWER(email))
     *   - Default role_id = 4 if missing
     *   - blocked: if empty or "0" => NULL; else => now() or parsed date
     *   - society_id: attach pivot; society_name: find (or optionally create)
     */
    public function importCsv(Request $request)
    {
        $data = $request->validate([
            'file'         => ['required', 'file', 'mimetypes:text/plain,text/csv,text/tsv,text/plain; charset=utf-8,text/any,text/*,application/csv,application/vnd.ms-excel'],
            'has_header'   => ['nullable','boolean'],
            'mode'         => ['required', Rule::in(['skip', 'update'])], // 'skip' -> ignore doublon, 'update' -> met à jour
            // optionnel: imposer un society_id pour tous
            'force_society_id' => ['nullable','integer','exists:societydb,id'],
        ]);

        $hasHeader  = (bool)($data['has_header'] ?? true);
        $mode       = $data['mode'];
        $forceSocId = $data['force_society_id'] ?? null;

        $file = $request->file('file')->getRealPath();

        $created = 0; $updated = 0; $skipped = 0;
        $errors  = [];

        // attendu (noms de colonnes – insensibles à la casse) :
        // firstname, lastname, email, password, phone, address1, address2, postcode, city, country_isocode, role_id, society_id
        $headerMap = [];

        if (($fh = fopen($file, 'r')) === false) {
            return back()->withErrors(['file' => 'Fichier illisible']).withInput();
        }

        $rowNum = 0;
        while (($row = fgetcsv($fh, 0, ',')) !== false) {
            $rowNum++;

            // lecture de l’entête
            if ($rowNum === 1 && $hasHeader) {
                $headerMap = $this->buildHeaderMap($row);
                continue;
            }

            // map colonnes -> valeurs
            $vals = $this->rowToAssoc($row, $headerMap);

            // normalisation email
            $email = Str::lower(trim($vals['email'] ?? ''));
            if ($email === '') {
                $errors[] = "Ligne {$rowNum}: email manquant";
                continue;
            }

            // autres champs
            $payload = [
                'firstname'       => trim($vals['firstname'] ?? ''),
                'lastname'        => trim($vals['lastname'] ?? ''),
                'phone'           => trim($vals['phone'] ?? ''),
                'address1'        => trim($vals['address1'] ?? ''),
                'address2'        => trim($vals['address2'] ?? ''),
                'postcode'        => trim($vals['postcode'] ?? ''),
                'city'            => trim($vals['city'] ?? ''),
                'country_isocode' => trim($vals['country_isocode'] ?? ''),
            ];

            // rôle: par défaut 4 (user pro), sauf si fourni
            $roleId = (int)($vals['role_id'] ?? 4);

            // mot de passe (si vide -> aléatoire)
            $pwdPlain = $vals['password'] ?? null;
            $password = $pwdPlain ? Hash::make($pwdPlain) : Hash::make(Str::random(16));

            // société (pivot)
            $societyId = $forceSocId ?: (isset($vals['society_id']) ? (int)$vals['society_id'] : null);
            if ($societyId && ! Society::whereKey($societyId)->exists()) {
                $errors[] = "Ligne {$rowNum}: society_id={$societyId} introuvable";
                $societyId = null; // on continue sans pivot
            }

            try {
                if ($mode === 'update') {
                    // upsert: met à jour si email existe, sinon crée
                    $user = User::updateOrCreate(
                        ['email' => $email],
                        array_merge($payload, [
                            'role_id'  => $roleId,
                            // si l’admin fournit un mot de passe, on l’applique même en update
                            'password' => $password,
                        ])
                    );
                    $user->wasRecentlyCreated ? $created++ : $updated++;

                } else {
                    // skip: si email existe -> on ignore
                    $existing = User::where('email', $email)->first();
                    if ($existing) {
                        $skipped++;
                        $user = $existing;
                    } else {
                        $user = User::create(array_merge($payload, [
                            'email'    => $email,
                            'role_id'  => $roleId,
                            'password' => $password,
                        ]));
                        $created++;

                        // envoie l’e-mail de vérification s’il est “MustVerifyEmail”
                        event(new Registered($user));
                        if (method_exists($user, 'sendEmailVerificationNotification')) {
                            $user->sendEmailVerificationNotification();
                        }
                    }
                }

                // pivot société : on crée le lien si society_id présent
                if ($societyId) {
                    // évite les doublons de pivot
                    $user->societies()->syncWithoutDetaching([$societyId]);
                }

            } catch (QueryException $e) {
                // 23000 = contrainte d’unicité / clé étrangère etc.
                $errors[] = "Ligne {$rowNum}: {$e->getMessage()}";
            } catch (\Throwable $e) {
                $errors[] = "Ligne {$rowNum}: {$e->getMessage()}";
            }
        }
        fclose($fh);

        return back()->with('ok', "Import terminé — créés: {$created}, mis à jour: {$updated}, ignorés: {$skipped}, erreurs: ".count($errors))
                     ->with('import_errors', $errors);
    }



    /**
     * Construit une map 'colName' => index à partir de la 1ère ligne
     */
    private function buildHeaderMap(array $header): array
    {
        $map = [];
        foreach ($header as $i => $name) {
            $key = Str::of($name)->trim()->lower()->replace([' ', '-'], ['_', '_'])->value();
            $map[$key] = $i;
        }
        return $map;
    }

    /**
     * Transforme une ligne CSV en tableau associatif basé sur $headerMap
     */
    private function rowToAssoc(array $row, array $map): array
    {
        // si pas d’entête: on suppose l’ordre “classique”
        if (empty($map)) {
            $keys = ['firstname','lastname','email','password','phone','address1','address2','postcode','city','country_isocode','role_id','society_id'];
            $out = [];
            foreach ($keys as $i => $k) {
                $out[$k] = $row[$i] ?? null;
            }
            return $out;
        }

        $out = [];
        foreach ($map as $k => $i) {
            $out[$k] = $row[$i] ?? null;
        }
        return $out;
    }

    
}
