<?php

    namespace App\Http\Controllers\Admin;

    use App\Http\Controllers\Controller;
    use App\Models\Point;
    use Illuminate\Http\Request;
    use Illuminate\Support\Facades\Http;
    use Illuminate\Support\Facades\Storage;
    use Illuminate\Support\Facades\Log;
    use Illuminate\Support\Str;
    use Illuminate\Support\Facades\Route;
    use Illuminate\Support\Facades\URL;
    use App\Services\PointQr;
    use App\Support\Serial; // helper trait if you enable that option
 




    class AdminWaypointController extends Controller
    {
        public function index(Request $request)
        {
            // Check authorizations first
            if (auth()->id() !== 1) {
                abort(403);
            }

            $q = $request->string('q');
        
            $points = Point::query()
                ->with(['user:id,firstname,lastname,email']) // eager load owner
                ->when($q, function ($query) use ($q) {
                    $like = '%'.$q.'%';
                    $query->where(function ($w) use ($like) {
                        $w->where('name', 'like', $like)
                        ->orWhere('city', 'like', $like)
                        ->orWhere('postcode', 'like', $like)
                        ->orWhere('address1', 'like', $like)
                        // Search by owner (first name, last name, email)
                        ->orWhereHas('user', function ($u) use ($like) {
                            $u->where('firstname', 'like', $like)
                                ->orWhere('lastname', 'like', $like)
                                ->orWhere('email', 'like', $like);
                        });
                    });
                })
                ->latest('id')
                ->paginate(20)
                ->withQueryString();
        
            return view('admin.points.index', compact('points','q'));
        }




        public function edit(Point $point)
        {
            return view('admin.points.edit', compact('point'));
        }






        public function update(Request $request, Point $point)
        {
            // 1) Validate input (photo must be an uploaded file, not a string)
            $validated = $request->validate([
                'name'            => 'required|string|max:255',
                'latitude'        => 'required|numeric',
                'longitude'       => 'required|numeric',
                'altitude'        => 'nullable|numeric',
                'floor'           => 'nullable|integer|min:0',
                'comment'         => 'nullable|string|max:1000',
                'address1'        => 'nullable|string|max:255',
                'address2'        => 'nullable|string|max:255',
                'postcode'        => 'nullable|string|max:255',
                'city'            => 'nullable|string|max:255',
                'country_isocode' => 'nullable|string|max:3',
                'favoris'         => 'nullable|in:0,1',
                'photo'           => 'nullable|image|max:5120', // 5 MB
            ]);
        
            // 2) Remove 'photo' before mass assignment to avoid storing the temporary path
            $data = collect($validated)->except('photo')->all();
            $point->fill($data);
        
            // 3) Upload if a file was provided
            if ($request->hasFile('photo')) {
                // Delete the existing one to avoid cluttering the disk
                if ($point->photo && Storage::disk('public')->exists($point->photo)) {
                    Storage::disk('public')->delete($point->photo);
                }
        
                // Store under storage/app/public/points/photos/...
                $path = $request->file('photo')->store('points/photos', 'public');
        
                // Persist the relative path in the database (e.g. points/photos/garage.png)
                $point->photo = $path;
            }
        
            $point->save();
        
            return back()->with('ok', __('Saved'));
            // Regenerate the QR if requested
            if ($request->boolean('regenerate_qr')) {
                PointQr::generateFor($point, true);
            }
    

            return redirect()->route('admin.points.index')->with('ok', 'Waypoint mis à jour.');
        }






        public function bulkFillAddresses(Request $request)
        {
            $max = (int)($request->input('max', 500));
            $count = 0;

            Point::where(function ($w) {
                $w->whereNull('address1')->orWhereNull('city')->orWhere('address1','')->orWhere('city','');
            })
            ->orderBy('id')
            ->chunkById(100, function ($chunk) use (&$count, $max) {
                foreach ($chunk as $point) {
                    if ($count >= $max) return false;

                    $addr = $this->reverseGeocodeArr($point->latitude, $point->longitude);
                    if (!empty($addr)) {
                        $point->update([
                            'address1'        => $addr['address1']        ?? $point->address1,
                            'address2'        => $addr['address2']        ?? $point->address2,
                            'postcode'        => $addr['postcode']        ?? $point->postcode,
                            'city'            => $addr['city']            ?? $point->city,
                            'country_isocode' => $addr['country_isocode'] ?? $point->country_isocode,
                        ]);
                        $count++;
                    }
                }
            });

            return back()->with('ok', "Adresses complétées pour $count waypoint(s).");
        }






        /**
         * Generate (or regenerate) every waypoint QR code.
         */
        // public function bulkGenerateQrcodes(Request $request)
        // {
        //     $locale = app()->getLocale(); // critical for the localized route
        //     $logo   = '/assets/images/V_300.svg';

        //     Point::select('id','name','serial','qrcode')
        //         ->orderBy('id')
        //         ->chunkById(200, function ($rows) use ($locale, $logo) {
        //             foreach ($rows as $p) {
        //                 // Generate a serial if missing
        //                 if (empty($p->serial)) {
        //                     $p->serial = \App\Http\Controllers\PointController::makeSerial($p);
        //                     $p->save();
        //                 }

        //                 try {
        //                     // URL publique (absolue)
        //                     // $url = route('points.public.readonly', [
        //                     //     'locale' => $locale,
        //                     //     'token'  => $p->serial,
        //                     // ], true);

        //                     $url = route('points.wx', ['token' => $p->serial], true);

        //                     // Generate the QR in SVG with the logo
        //                     $svg = PointQr::makeSvgWithLogo($url, $logo, 512, 0.18);

        //                     // Nom du fichier
        //                     $filename = "V{$p->id}-{$p->serial}.svg";
        //                     $path     = "qrcodes/{$filename}"; // stockage dans public/qrcodes/

        //                     // Sauvegarde avec Storage (remplace file_put_contents)
        //                     Storage::disk('public')->put($path, $svg);

        //                     // Sauvegarde chemin en DB
        //                     $p->qrcode = $path;
        //                     $p->save();
        //                 } catch (\Throwable $e) {
        //                     Log::error('QR generation failed', [
        //                         'point_id' => $p->id,
        //                         'err'      => $e->getMessage(),
        //                     ]);
        //                 }
        
        //             }
        //         });

        //     return back()->with('ok', 'QR codes generated and stored in public/qrcodes/.');
        // }



        public function bulkGenerateQrcodes(\Illuminate\Http\Request $request)
        {
            \App\Models\Point::select('id','name','serial','qrcode')
                ->orderBy('id')
                ->chunkById(200, function ($rows) {
                    foreach ($rows as $p) {
                        \App\Services\PointQr::generateSvgSimple($p);
                    }
                });
        
            return back()->with('ok', 'QR codes (simples) générés.');
        }






        // ---------- helpers ----------
        private function reverseGeocodeArr($lat, $lng): array
        {
            try {
                $ua = config('app.name','Laravel').'/'.app()->version();
                $email = config('mail.from.address');

                $res = Http::withHeaders(['User-Agent' => $ua])
                    ->timeout(8)
                    ->get('https://nominatim.openstreetmap.org/reverse', [
                        'format'         => 'jsonv2',
                        'lat'            => $lat,
                        'lon'            => $lng,
                        'zoom'           => 18,
                        'addressdetails' => 1,
                        'email'          => $email,
                    ]);

                if (!$res->ok()) return [];

                $j = $res->json();
                $a = $j['address'] ?? [];

                $address1 = trim(($a['house_number'] ?? '').' '.($a['road'] ?? ''));
                $address2 = $a['suburb'] ?? ($a['neighbourhood'] ?? null);
                $city     = $a['city'] ?? $a['town'] ?? $a['village'] ?? $a['hamlet'] ?? null;
                $postcode = $a['postcode'] ?? null;
                $cc       = strtoupper($a['country_code'] ?? '');

                return [
                    'address1'        => $address1 ?: null,
                    'address2'        => $address2,
                    'postcode'        => $postcode,
                    'city'            => $city,
                    'country_isocode' => $cc ?: null,
                ];
            } catch (\Throwable $e) {
                return [];
            }
        }




        public function generateSerials(Request $request)
        {
            $updated = 0;
        
            Point::whereNull('serial')->orWhere('serial','')
                ->chunkById(500, function ($chunk) use (&$updated) {
                    foreach ($chunk as $point) {
                        Serial::ensure($point); // create a URL-safe serial according to the helper
                        $updated++;
                    }
                });
        
            return back()->with('status', $updated
                ? "{$updated} serial(s) généré(s)."
                : "Aucun serial à générer : tout est déjà rempli.");
        }




        
        public function generateMissingQr()
        {
            $locale = app()->getLocale();
            $logo   = '/assets/images/V_300.svg';

            \App\Models\Point::select('id','name','serial','qrcode')
                ->whereNull('qrcode')
                ->orWhere('qrcode', '')
                ->orderBy('id')
                ->chunkById(200, function ($rows) use ($locale, $logo) {
                    foreach ($rows as $p) {
                        if (empty($p->serial)) {
                            $p->serial = \App\Http\Controllers\PointController::makeSerial($p);
                            $p->save();
                        }

                        try {
                            $url = route('points.public.readonly', [
                                'locale' => $locale,
                                'token'  => $p->serial,
                            ], true);

                            $svg = PointQr::makeSvgWithLogo($url, $logo, 512, 0.18);

                            $relPath = "qrcodes/V{$p->id}-{$p->serial}.svg";
                            Storage::disk('public')->put($relPath, $svg);

                            $p->qrcode = $relPath;
                            $p->save();
                        } catch (\Throwable $e) {
                            Log::error('QR generation failed', [
                                'point_id' => $p->id,
                                'err'      => $e->getMessage(),
                            ]);
                        }
                    }
                });

            return back()->with('ok', 'QR codes (re)générés pour les manquants.');
        }


    }
