<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Auth\Events\Registered;
use Illuminate\Database\QueryException;
use Illuminate\Validation\Rule;


class AuthController extends Controller
{
   
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required'
        ]);

        if (Auth::attempt($request->only('email', 'password'))) {
            $request->session()->regenerate();
            return redirect()->intended(lr('search'));
        }

        return back()->withErrors([
            'email' => 'Les identifiants ne correspondent pas.',
        ]);
    }





    public function register(Request $request)
    {

        $request->merge(['email' => \Illuminate\Support\Str::lower(trim($request->input('email')))]);

        $data = $request->validate([
            'firstname' => ['required','string','max:100'],
            'lastname'  => ['required','string','max:100'],
            'phone'     => ['nullable','string','max:50'],
            'address1'  => ['nullable','string','max:255'],
            'address2'  => ['nullable','string','max:255'],
            'postcode'  => ['nullable','string','max:50'],
            'city'      => ['nullable','string','max:150'],
            'country_isocode' => ['nullable','string','max:2'],
            'email'     => ['required','email:rfc,dns','max:255', Rule::unique('users','email')->whereNull('blocked')],
            'password'  => ['required','string','min:8','confirmed'],
        ]);
    

        // (4) existe mais non vérifié ?
        $existing = \App\Models\User::where('email', $data['email'])->first();
        if ($existing) {
            if (method_exists($existing, 'hasVerifiedEmail') && $existing->hasVerifiedEmail()) {
                return back()->withErrors(['email'=>__('This email is already registered. Please log in.')])
                            ->withInput($request->except('password','password_confirmation'));
            } else {
                if (method_exists($existing, 'sendEmailVerificationNotification')) {
                    $existing->sendEmailVerificationNotification();
                }
                return redirect()->route('verification.notice')
                    ->with('ok', __('We re-sent the verification email to :email.', ['email'=>$existing->email]));
            }
        }


        // création
        
        try {
            $user = \App\Models\User::create([
                'firstname' => $data['firstname'],
                'lastname'  => $data['lastname'],
                'phone'     => $data['phone'] ?? null,
                'address1'  => $data['address1'] ?? null,
                'address2'  => $data['address2'] ?? null,
                'postcode'  => $data['postcode'] ?? null,
                'city'      => $data['city'] ?? null,
                'country_isocode' => $data['country_isocode'] ?? null,
                'email'     => $data['email'],
                'password'  => bcrypt($data['password']),
            ]);
        } catch (QueryException $e) {
            if ((int)$e->getCode() === 23000) {
                return back()
                    ->withErrors(['email' => __('This email is already in use. Try logging in or resetting your password.')])
                    ->withInput($request->except('password','password_confirmation'));
            }
            throw $e;
        }

        // Déclenche les listeners Laravel (dont l’email de vérification)
        event(new Registered($user));

        //Auth::login($user);

        // if ($user instanceof \Illuminate\Contracts\Auth\MustVerifyEmail) {
        //     $user->sendEmailVerificationNotification();
        // }

       
        // $isSuper = (int)auth()->id() === 1 || (int)(auth()->user()->role_id ?? 0) === 1; 
        // if ($isSuper) {
        //     return redirect()->route('admin.points.index', ['locale' => app()->getLocale()]); 
        // }        

        return redirect()->route('verification.notice', ['locale' => app()->getLocale()])
        ->with('ok', __('We sent you a verification email. Please check your inbox.'));
        
        
    }







    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect(lr('/'));
    }
}