<?php

// app/Http/Controllers/ItineraryController.php
namespace App\Http\Controllers;

use App\Models\Itinerary;
use App\Models\Point;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Carbon\Carbon;
use App\Models\ItineraryPoint;
use App\Models\ItineraryPointUser;
use Illuminate\Validation\Rule;
use App\Services\ItinerarySyncer;
use Illuminate\Support\Arr;

class ItineraryController extends Controller
{


    
    // ------------------------
    // Index
    // ------------------------
    public function index(Request $request, ?string $locale = null)
    {
        $user = $request->user();
        $locale = $locale ?: app()->getLocale();

        Gate::authorize('viewAny', Itinerary::class);

        $query = Itinerary::with(['assignedUsers:id,firstname,lastname,email','points:id,name'])
            ->orderByDesc('created_at');

        if ($user->id !== 1) {
            $societyId = $user->societies()->value('society_id');
            $query->where('society_id', $societyId);
        }

        $itineraries = $query->paginate(20);

        return view('itins.index', compact('itineraries'));
    }





    // ------------------------
    // Create
    // ------------------------
    public function create(Request $request, ?string $locale = null)
    {
        $user = $request->user();
        $locale = $locale ?: app()->getLocale();

        Gate::authorize('create', Itinerary::class);

        if ($user->id === 1) {
            // Admin : toutes les sociétés qui ont au moins un admin pro
            $societies = Society::whereHas('users', function($q){
                $q->where('role_id', 2); // admin pro
            })
            ->orderBy('name')
            ->get(['societydb.id','societydb.name']);
        } else {
            // Admin Pro : seulement ses sociétés
            $societies = $user->societies()->orderBy('name')->get(['id','name']);
        }

         
        return view('itins.create', [
            'societies' => $societies,
            'isSuper'   => $user->id === 1,
        ]);
    }





    // ------------------------
    // Store
    // ------------------------
    public function store(Request $request, string $locale = 'fr')
    {
        $data = $request->validate([
            'name'              => ['required','string','max:255','unique:itineraries,name'],
            'society_id'        => ['required','integer','exists:societydb,id'],
            'assigned_user_id'  => ['required','integer','exists:users,id'], // admin pro responsable
            'assigned_user_ids' => ['array'],
            'assigned_user_ids.*' => ['integer','exists:users,id'],
            'point_ids'         => ['array','min:1'],
            'point_ids.*'       => ['integer','exists:points,id'],
            // champs horaires/commentaires optionnels, indexés comme point_ids
            'planned_at'        => ['array'],
            'arrival_at'        => ['array'],
            'validated_at'      => ['array'],
            'rescheduled_at'    => ['array'],
            'comment'           => ['array'],
            'done'              => ['array'],
        ]);
    
        $assignedUserIds = array_values(array_unique($data['assigned_user_ids'] ?? []));
        if (!in_array($data['assigned_user_id'], $assignedUserIds, true)) {
            $assignedUserIds[] = (int)$data['assigned_user_id']; // inclure l’admin pro désigné
        }
    
        $pointIds   = $data['point_ids'] ?? [];
        $plannedA   = $data['planned_at'] ?? [];
        $arrivalA   = $data['arrival_at'] ?? [];
        $validA     = $data['validated_at'] ?? [];
        $replanA    = $data['rescheduled_at'] ?? [];
        $commentA   = $data['comment'] ?? [];
        $doneA      = $data['done'] ?? [];
    
        $user = $request->user();
    
        return DB::transaction(function () use ($data, $user, $assignedUserIds, $pointIds, $plannedA, $arrivalA, $validA, $replanA, $commentA, $doneA) {
    
            $it = Itinerary::create([
                'name'            => $data['name'],
                'society_id'      => $data['society_id'],
                'owner_user_id'   => $user->id,              // créateur
                'assigned_user_id'=> $data['assigned_user_id'], // responsable (admin pro)
                'scheduled_date'  => $data['scheduled_date'] ?? null,
            ]);
    
            // Pivot principal users ←→ itinerary
            if (!empty($assignedUserIds)) {
                $it->assignedUsers()->sync($assignedUserIds);
            }
    
            // 1) Écrire itinerary_points (en respectant l’ordre et les doublons)
            $items = [];
            foreach ($pointIds as $idx => $pid) {
                $items[] = new ItineraryPoint([
                    'point_id'      => (int)$pid,
                    'sort_order'    => $idx + 1,
                    'planned_at'    => Arr::get($plannedA, $idx) ?: null,
                    'arrival_at'    => Arr::get($arrivalA, $idx) ?: null,
                    'done'          => (int)Arr::get($doneA, $idx, 0) === 1,
                    'validated_at'  => Arr::get($validA, $idx) ?: null,
                    'rescheduled_at'=> Arr::get($replanA, $idx) ?: null,
                    'comment'       => Arr::get($commentA, $idx) ?: null,
                ]);
            }
            if (!empty($items)) {
                $it->items()->saveMany($items);
            }
    
            // 2) Seed initial dans itinerary_points_user pour CHAQUE user assigné
            if (!empty($assignedUserIds) && !empty($items)) {
                $rows = [];
                foreach ($assignedUserIds as $uid) {
                    foreach ($items as $ip) {
                        $rows[] = [
                            'itinerary_id' => $it->id,
                            'user_id'      => (int)$uid,
                            'point_id'     => $ip->point_id,
                            'sort_order'   => $ip->sort_order,
                            'planned_at'   => $ip->planned_at,
                            'arrival_at'   => $ip->arrival_at,
                            'done'         => $ip->done,
                            'validated_at' => $ip->validated_at,
                            'rescheduled_at'=> $ip->rescheduled_at,
                            'comment'      => $ip->comment,
                            'created_at'   => now(),
                            'updated_at'   => now(),
                        ];
                    }
                }
                ItineraryPointUser::insert($rows);
            }
    
            return redirect()
                ->route('admin.itins.edit', ['locale'=>$locale, 'itinerary'=>$it->id])
                ->with('ok', __('Itinerary created.'));
        });
    }





    // ------------------------
    // Edit
    // ------------------------
    public function edit(Request $request, string $locale, Itinerary $itinerary)
    {
        $user = auth()->user();
        //$locale = $request->route('locale'); //$locale ?: app()->getLocale();

  
        // Signature supports both groups; when called from admin (non localized),
        // Laravel will pass (Itinerary $itinerary, Request $request)
        // We normalize parameters here:
        if ($itinerary instanceof Request) {
            // swapped (admin mode)
            $request   = $itinerary;
            $itinerary = func_get_arg(0); // first arg is actually Itinerary
        }

        
        $isSuper= (int)$user->id === 1;

        if (!$isSuper) {
            $socId = $user->societies()->value('society_id');
            abort_if((int)$itinerary->society_id !== (int)$socId, 403);
        }

        // Societies list
        // $societies = $isSuper
        //     ? Society::whereHas('users', fn($q)=>$q->where('role_id',2))->orderBy('name')->get(['id','name'])
        //     : $user->societies()->orderBy('name')->get(['id','name']);

        $societies = $isSuper
            ? \App\Models\Society::query()
                ->select('societydb.id as id', 'societydb.name')
                ->whereHas('users', fn ($q) => $q->where('role_id', 2))
                ->orderBy('societydb.name')
                ->get()
            : $user->societies()   // belongsToMany -> joins societydb_user => must qualify
                ->select('societydb.id as id', 'societydb.name')
                ->orderBy('societydb.name')
                ->get();

        // Assigned users (preselected)
        $preUsers = $itinerary->assignedUsers()->pluck('users.id')->all();

        // Selected points (ordered)
        $selectedPoints   = $itinerary->points()->get(['points.id','points.name','points.latitude','points.longitude']);
        $selectedPointIds = $selectedPoints->pluck('id')->all();

        return view('itins.edit', compact('itinerary','societies','preUsers','selectedPoints','selectedPointIds'));
    }





    // ------------------------
    // Update
    // ------------------------
    public function update(Request $request, string $locale, Itinerary $itinerary)
    {
        $data = $request->validate([
            'name'              => ['required','string','max:255','unique:itineraries,name,'.$itinerary->id],
            'society_id'        => ['required','integer','exists:societydb,id'],
            'assigned_user_id'  => ['required','integer','exists:users,id'],
            'assigned_user_ids' => ['array'],
            'assigned_user_ids.*' => ['integer','exists:users,id'],
            'point_ids'         => ['array','min:1'],
            'point_ids.*'       => ['integer','exists:points,id'],
            'planned_at'        => ['array'],
            'arrival_at'        => ['array'],
            'validated_at'      => ['array'],
            'rescheduled_at'    => ['array'],
            'comment'           => ['array'],
            'done'              => ['array'],
        ]);

        $assignedUserIds = array_values(array_unique($data['assigned_user_ids'] ?? []));
        if (!in_array($data['assigned_user_id'], $assignedUserIds, true)) {
            $assignedUserIds[] = (int)$data['assigned_user_id'];
        }

        $pointIds = $data['point_ids'] ?? [];
        $plannedA = $data['planned_at'] ?? [];
        $arrivalA = $data['arrival_at'] ?? [];
        $validA   = $data['validated_at'] ?? [];
        $replanA  = $data['rescheduled_at'] ?? [];
        $commentA = $data['comment'] ?? [];
        $doneA    = $data['done'] ?? [];

        return DB::transaction(function () use ($itinerary, $data, $assignedUserIds, $pointIds, $plannedA, $arrivalA, $validA, $replanA, $commentA, $doneA, $locale) {

            // Maj de base
            $itinerary->fill([
                'name'             => $data['name'],
                'society_id'       => $data['society_id'],
                'assigned_user_id' => $data['assigned_user_id'],
                'scheduled_date'   => $data['scheduled_date'] ?? $itinerary->scheduled_date,
            ])->save();

            // Sync des users assignés
            $existingUserIds = $itinerary->assignedUsers()->pluck('users.id')->all();
            $itinerary->assignedUsers()->sync($assignedUserIds);
            $addedUserIds   = array_values(array_diff($assignedUserIds, $existingUserIds));
            $removedUserIds = array_values(array_diff($existingUserIds, $assignedUserIds));

            // Rebuild complet de itinerary_points selon la sélection actuelle (garde les doublons)
            $itinerary->items()->delete();
            $items = [];
            foreach ($pointIds as $idx => $pid) {
                $items[] = new ItineraryPoint([
                    'point_id'      => (int)$pid,
                    'sort_order'    => $idx + 1,
                    'planned_at'    => Arr::get($plannedA, $idx) ?: null,
                    'arrival_at'    => Arr::get($arrivalA, $idx) ?: null,
                    'done'          => (int)Arr::get($doneA, $idx, 0) === 1,
                    'validated_at'  => Arr::get($validA, $idx) ?: null,
                    'rescheduled_at'=> Arr::get($replanA, $idx) ?: null,
                    'comment'       => Arr::get($commentA, $idx) ?: null,
                ]);
            }
            if (!empty($items)) {
                $itinerary->items()->saveMany($items);
            }

            // 🔁 Seed / MAJ côté users
            // 1) Supprimer les lignes des users retirés
            if (!empty($removedUserIds)) {
                ItineraryPointUser::where('itinerary_id', $itinerary->id)
                    ->whereIn('user_id', $removedUserIds)
                    ->delete();
            }

            // 2) Pour les users existants : on NE TOUCHE PAS à leurs progrès (non destructif)
            //    On ne va insérer que les nouveaux points (si l’itinéraire a gagné des points).
            //    Pour simplifier ici, on repart propre si tu préfères : décommente le bloc “reset tout”.
            //
            // --- reset tout (optionnel) ---
            // ItineraryPointUser::where('itinerary_id', $itinerary->id)
            //     ->whereIn('user_id', $assignedUserIds)
            //     ->delete();
            // $seedUsers = $assignedUserIds;
            //
            // --- non destructif (par défaut) ---
            $seedUsers = $addedUserIds; // on seed seulement pour les nouveaux users

            // 3) Seed pour $seedUsers (nouveaux users) : copie des items actuels
            if (!empty($seedUsers) && !empty($items)) {
                $rows = [];
                foreach ($seedUsers as $uid) {
                    foreach ($items as $ip) {
                        $rows[] = [
                            'itinerary_id' => $itinerary->id,
                            'user_id'      => (int)$uid,
                            'point_id'     => $ip->point_id,
                            'sort_order'   => $ip->sort_order,
                            'planned_at'   => $ip->planned_at,
                            'arrival_at'   => $ip->arrival_at,
                            'done'         => $ip->done,
                            'validated_at' => $ip->validated_at,
                            'rescheduled_at'=> $ip->rescheduled_at,
                            'comment'      => $ip->comment,
                            'created_at'   => now(),
                            'updated_at'   => now(),
                        ];
                    }
                }
                if (!empty($rows)) {
                    ItineraryPointUser::insert($rows);
                }
            }

            // 4) Si l’itinéraire a des NOUVEAUX points et que tu veux les pousser à tous les users existants
            //    sans casser leurs dates/“done” actuels, on insère uniquement les (itinerary_id, user_id, sort_order)
            //    qui n’existent pas encore (requête d’upsert ciblée). Voici une version simple :
            if (!empty($items) && !empty($assignedUserIds)) {
                foreach ($assignedUserIds as $uid) {
                    // récupère les orders déjà présents
                    $presentOrders = ItineraryPointUser::where('itinerary_id',$itinerary->id)
                        ->where('user_id',$uid)
                        ->pluck('sort_order')->all();
                    $toInsert = [];
                    foreach ($items as $ip) {
                        if (in_array($ip->sort_order, $presentOrders, true)) continue;
                        $toInsert[] = [
                            'itinerary_id'=>$itinerary->id,
                            'user_id'=>$uid,
                            'point_id'=>$ip->point_id,
                            'sort_order'=>$ip->sort_order,
                            'planned_at'=>$ip->planned_at,
                            'arrival_at'=>$ip->arrival_at,
                            'done'=>$ip->done,
                            'validated_at'=>$ip->validated_at,
                            'rescheduled_at'=>$ip->rescheduled_at,
                            'comment'=>$ip->comment,
                            'created_at'=>now(),
                            'updated_at'=>now(),
                        ];
                    }
                    if (!empty($toInsert)) {
                        ItineraryPointUser::insert($toInsert);
                    }
                }
            }

            return redirect()
                ->route(request()->routeIs('pro.*') ? 'pro.itins.edit' : 'admin.itins.edit', [
                    'locale'=>$locale,
                    'itinerary'=>$itinerary->id
                ])->with('ok', __('Itinerary updated.'));
        });
    }
    // public function update(Request $request, string $locale, Itinerary $itinerary)
    // {
    //     // Récupération des points envoyés depuis le formulaire
    //     $ids = array_values((array) $request->input('point_ids', []));

    //     DB::transaction(function () use ($itinerary, $ids) {
    //         // Supprime les anciens items
    //         $itinerary->items()->delete();

    //         // Réinsère les nouveaux (avec doublons autorisés)
    //         $now = now();
    //         $rows = [];
    //         foreach ($ids as $i => $pointId) {
    //             $rows[] = [
    //                 'itinerary_id' => $itinerary->id,
    //                 'point_id'     => (int) $pointId,
    //                 'sort_order'   => $i + 1,
    //                 'created_at'   => $now,
    //                 'updated_at'   => $now,
    //             ];
    //         }

    //         if ($rows) {
    //             ItineraryPoint::insert($rows);
    //         }
    //     });

    //     return redirect()->back()->with('ok', __('Itinerary updated successfully.'));
    // }




    // ------------------------
    // Destroy
    // ------------------------
    public function destroy(Request $request, string $locale, Itinerary $itinerary)
    {
        $user = auth()->user();
        $locale = $locale ?: app()->getLocale();

        $isSuper = (int)$user->id === 1;

        if (!$isSuper) {
            $socId = $user->societies()->value('society_id');
            abort_if((int)$itinerary->society_id !== (int)$socId, 403);
        }

        $itinerary->delete();

        return back()->with('ok', __('Itinerary deleted.'));
    }




    // ------------------------
    // Map 
    // ------------------------
    public function map(Request $request, string $locale, Itinerary $itinerary)
   {
        $user = $request->user();
        $locale = $locale ?: app()->getLocale();

        if ($itinerary instanceof Request) {
            $request   = $itinerary;
            $itinerary = func_get_arg(0);
        }

        $isSuper = (int)$user->id === 1;
        if (!$isSuper) {
            $socId = $user->societies()->value('society_id');
            abort_if((int)$itinerary->society_id !== (int)$socId, 403);
        }

        $points = $itinerary->points()->get(['points.id','points.name','points.latitude','points.longitude']);
        return view('itins.map', compact('itinerary','points'));
    }
    



     // ------------------------
    // AJAX: pro admins for a society
    // ------------------------
    public function adminsForSociety(Request $request, ?string $locale = null)
    {
        $user = auth()->user();
        $locale = $locale ?: app()->getLocale();

        $sid = (int) $request->query('society_id', 0);
        abort_if($sid <= 0, 400, 'Missing society_id');

        $admins = User::whereHas('societies', fn($q)=>$q->where('society_id',$sid))
            ->where('role_id', 2)
            ->orderBy('lastname')
            ->get(['id','firstname','lastname','email']);

        return response()->json($admins);
    }




    // ------------------------
    // AJAX: users for a society (id > 1)
    // ------------------------
    public function usersForSociety(Request $request, ?string $locale = null)
    {
        
        $locale = $locale ?: app()->getLocale();

        $sid = (int) $request->query('society_id', 0);
        abort_if($sid <= 0, 400, 'Missing society_id');

        $users = User::whereHas('societies', fn($q)=>$q->where('society_id',$sid))
            ->where('id', '>', 1)
            ->orderBy('lastname')
            ->get(['id','firstname','lastname','email']);

        return response()->json($users);
    }





    // ------------------------
    // AJAX: points for (society + admin) context
    // includes: points owned by society users + points shared to the chosen admin
    // ------------------------
    public function pointsForContext(Request $request, ?string $locale = null)
    {
        try {
            $data = $request->validate([
                'society_id' => ['required','integer','min:1'], // on garde le param pour cohérence UI, mais on ne filtre plus dessus
                'admin_id'   => ['required','integer','min:1'],
            ]);
            $adminId = (int) $data['admin_id'];
    
            // 1) IDs des points explicitement partagés à cet admin
            $sharingTable = Schema::hasTable('sharings')
                ? 'sharings'
                : (Schema::hasTable('sharingdb') ? 'sharingdb' : null);
    
            $sharedPointIds = [];
            if ($sharingTable) {
                $sharedPointIds = DB::table($sharingTable)
                    ->where('target_user_id', $adminId)
                    ->pluck('point_id')
                    ->filter()
                    ->map(fn($v)=>(int)$v)
                    ->values()
                    ->all();
            }
    
            // 2) Points appartenant AU PRO ADMIN (owner = admin_id)
            //    + points partagés à cet admin
            $q = Point::query()->select('id','name')
                ->where('user_id', $adminId);
    
            if (!empty($sharedPointIds)) {
                $q->orWhereIn('id', $sharedPointIds);
            }
    
            $points = $q->orderBy('name')->get()->unique('id')->values();
    
            return response()->json($points, 200);
        } catch (\Throwable $e) {
            Log::error('pointsForContext strict error', [
                'msg'  => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
            ]);
            // on renvoie un tableau vide pour ne pas casser l’UI
            return response()->json([], 200);
        }
    }
    


    // ------------------------
    // AJAX: sauvegarde de progression
    // ------------------------

    public function saveProgress(Request $request, string $locale, Itinerary $itinerary)
    {
        $user = $request->user();

        // Autorisations minimales : super-admin (id=1), owner, manager, ou user assigné.
        $isSuper   = (int)($user->role_id ?? 0) === 1 || (int)$user->id === 1;
        $isOwner   = (int)$itinerary->owner_user_id    === (int)$user->id;
        $isManager = (int)$itinerary->assigned_user_id === (int)$user->id;
        $isAssigned= $itinerary->assignedUsers()->where('users.id', $user->id)->exists();
        $isPro     = (int)($user->role_id ?? 0) === 2;
 
        $sameSoc   = $isPro
        ? $user->societies()->where('society_id', $itinerary->society_id)->exists()
        : false;


        if (!($isSuper || $isOwner || $isManager || $isAssigned)) {
            abort(403, 'forbidden');
        }

        // items = tableau : [ item_id => [done?, arrival_at?, validated_at?, rescheduled_at?, comment?], ... ]
        $data = $request->validate([
            'items'                          => ['required','array'],
            'items.*.done'                   => ['nullable'],        // checkbox
            'items.*.planned_at'             => ['nullable','date'],
            'items.*.arrival_at'             => ['nullable','date'],
            'items.*.validated_at'           => ['nullable','date'],
            'items.*.rescheduled_at'         => ['nullable','date'],
            'items.*.comment'                => ['nullable','string'],
        ]);

        // Helper strict pour le format <input type="datetime-local">
        $parseLocal = function (?string $v): ?Carbon {
            if (!$v) return null;
            try {
                return Carbon::createFromFormat('Y-m-d\TH:i', $v, config('app.timezone'))->seconds(0);
            } catch (\Throwable $e) {
                // fallback permissif si le navigateur envoie autre chose
                return Carbon::parse($v);
            }
        };

        DB::transaction(function () use ($data, $itinerary, $parseLocal) {
            foreach ($data['items'] as $itemId => $fields) {
                // Sécurité : ne mettre à jour que les items appartenant à CET itinéraire
                $item = $itinerary->items()->whereKey($itemId)->first();
                if (!$item) {
                    continue; // item étranger → ignoré
                }

                $item->done           = array_key_exists('done', $fields) ? 1 : 0;
                $item->planned_at     = $parseLocal($fields['planned_at']    ?? null);
                $item->arrival_at     = $parseLocal($fields['arrival_at']    ?? null);
                $item->validated_at   = $parseLocal($fields['validated_at']  ?? null);
                $item->rescheduled_at = $parseLocal($fields['rescheduled_at']?? null);
                $item->comment        = $fields['comment'] ?? null;

                $item->save();
            }
        });

        // Réponse selon le contexte (form classique ou XHR/fetch)
        if ($request->expectsJson()) {
            return response()->json(['ok' => true]);
        }

        return back()->with('ok', __('Progress saved.'));
    }



    public function validateItinerary(Request $request, string $locale, Itinerary $itinerary)
    {
        // Autorisation (à adapter à ta logique)
        if (Gate::denies('manage-itinerary', $itinerary)) {
            abort(403);
        }
    
        // Marque l’itinéraire comme validé (champ à adapter: validated_date / validated_at)
        $itinerary->validated_date = Carbon::now();
        $itinerary->save();
    
        // Optionnel: si tu passes aussi l’état des points depuis la page carte,
        // tu peux boucler ici sur $request->input('points', []) pour MAJ itinerary_points.
    
        return back()->with('ok', __('Itinerary validated.'));
    }
  
}