<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;

use App\Models\User;
use App\Models\Sharing;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Gate;
use App\Models\Society; 

use App\Models\Point;
use Symfony\Component\Mailer\Exception\TransportExceptionInterface;

use SimpleSoftwareIO\QrCode\Facades\QrCode;

use App\Services\PointQr;
use App\Mail\PointSharedMail;





class PointController extends Controller
{


    public function home_index()
    {
        $coord = Point::all();
        return view('welcome', compact('coord'));
    }






    public function register_index()
    {
        $coord = Point::all();
        return view('register', compact('coord'));
    }






    /** Protected list of waypoints */
    public function point_index()
    {
        $user = auth()->user();

        if ($user && $user->id === 1) {
            // Super admin can see everything
            $points = Point::with('owner:id,firstname,lastname,email')->latest()->paginate(20);
            return view('points.index', compact('points'));
        }

        $points = Point::with('owner:id,firstname,lastname,email')
            ->when($user, fn ($q) => $q->visibleFor($user))
            ->latest()
            ->paginate(20);

        return view('points.index', compact('points'));
    }






    /** Protected map view */
    public function displayMap_index()
    {
        $user = auth()->user();

        $points = Point::select([
                'id','user_id','name','latitude','longitude','altitude','floor','favoris',
                'comment','address1','address2','postcode','city','country_isocode',
                'photo','serial','qrcode'
            ])
            ->when($user, fn ($q) => $q->visibleFor($user)) // only when a user is present
            ->latest()
            ->get();

        return view('map', compact('points'));
    }






    /** Protected create action */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'             => ['required','string','max:255','unique:points,name'],
            'latitude'         => 'required|numeric',
            'longitude'        => 'required|numeric',
            'altitude'         => 'nullable|numeric',
            'floor'            => 'nullable|integer|min:0',
            'comment'          => 'nullable|string|max:1000',
            'address1'         => 'nullable|string|max:255',
            'address2'         => 'nullable|string|max:255',
            'postcode'         => 'nullable|string|max:255',
            'city'             => 'nullable|string|max:255',
            'country_isocode'  => 'nullable|string|max:3',
            'favoris'          => 'nullable|in:0,1',
            'photo'            => 'nullable|image|max:5120', // 5 MB
        ]);

        $photoFile = $request->file('photo');
        unset($data['photo']);

        $point = new Point($data);
        $point->user_id = auth()->id() ?: 0;
        $point->floor = $request->filled('floor') ? (int) $request->input('floor') : 0;
        $point->favoris = isset($data['favoris']) ? (int) $data['favoris'] : 0;

        if ($photoFile) {
            $point->photo = Storage::disk('public')->putFile('points/photos', $photoFile);
        }

        $point->save();

        if (empty($point->serial)) {
            $point->serial = static::makeSerial($point);
            $point->save();
        }

        try {
            $locale = app()->getLocale();
            $url = route('points.public.readonly', [
                'locale' => $locale,
                'token'  => $point->serial,
            ], true);

  
            // Path within public storage (storage/app/public/qrcodes)
            $filename = "V{$point->id}-{$point->serial}.svg";
            $qrRelativePath = "qrcodes/{$filename}";

            Storage::disk('public')->makeDirectory('qrcodes');
            $logoPublicPath = public_path('assets/images/V_300.svg'); // or .png
            $svg = PointQr::makeSvgWithLogo($url, $logoPublicPath, 512, 0.18);
            Storage::disk('public')->put($qrRelativePath, $svg);

            $point->qrcode = $qrRelativePath;
            $point->save();

        } catch (\Throwable $e) {
            Log::error('QR generation failed on store', ['point_id'=>$point->id, 'err'=>$e->getMessage()]);
            // Do not block the creation flow
        }

     
        return redirect()->route('search', ['locale' => app()->getLocale()])
        ->with('ok', __('Waypoint created'));
    }




    

  


    public static function makeSerial(Point $point): string
    {
        $id   = $point->id;
        $date = $point->created_at ? $point->created_at->format('Y-m-d') : now()->format('Y-m-d');
    
        $raw = "{$id}-{$date}";
        // Base64 URL-safe, no padding
        return rtrim(strtr(base64_encode($raw), '+/', '-_'), '=');
    }







    /** Protected update */
    public function update(Request $request, Point $point)
    {
        if (auth()->id() !== 1 && $point->user_id !== auth()->id()) {
            abort(403);
        }
        
        $data = $request->validate([
            'name'             => ['required','string','max:255', Rule::unique('points','name')->ignore($point->id)],
            'latitude'         => 'required|numeric',
            'longitude'        => 'required|numeric',
            'altitude'         => 'nullable|numeric',
            'floor'            => 'nullable|integer|min:0',
            'comment'          => 'nullable|string|max:1000',
            'address1'         => 'nullable|string|max:255',
            'address2'         => 'nullable|string|max:255',
            'postcode'         => 'nullable|string|max:255',
            'city'             => 'nullable|string|max:255',
            'country_isocode'  => 'nullable|string|max:3',
            'favoris'          => 'nullable|in:0,1',
            'photo'            => 'nullable|image|max:5120',
        ]);

        $photoFile = $request->file('photo');
        unset($data['photo']);

        $point->fill($data);
        $point->floor = $request->filled('floor') ? (int) $request->input('floor') : 0;
        $point->favoris = isset($data['favoris']) ? (int) $data['favoris'] : $point->favoris;

        // Generate a serial if it is missing
        if (empty($point->serial)) {
            $point->serial = static::makeSerial($point);
        }

        // Photo: replace the previous file if a new one was uploaded
        if ($photoFile) {
            // Optionally delete the previous file
            if (!empty($point->photo)) {
                Storage::disk('public')->delete($point->photo);
            }
            $path = Storage::disk('public')->putFile('points/photos', $photoFile);
            $point->photo = $path;
        }

        
        
        // (Re)generate the QR if it does not exist yet—or if you want to force regeneration
        if (empty($point->qrcode)) {
            try {
                $locale = app()->getLocale();
                $url = route('points.public.readonly', [
                    'locale' => $locale,
                    'token'  => $point->serial,
                ], true);


                // Path within public storage (storage/app/public/qrcodes)
                Storage::disk('public')->makeDirectory('qrcodes');
                $filename = "V{$point->id}-{$point->serial}.svg";
                $qrRelativePath = "qrcodes/{$filename}";
                $logoPublicPath = public_path('assets/images/V_300.svg'); // or .png
                $svg = PointQr::makeSvgWithLogo($url, $logoPublicPath, 512, 0.18);
                Storage::disk('public')->put($qrRelativePath, $svg);
                $point->qrcode = $qrRelativePath;
                $point->save();

            } catch (\Throwable $e) {
                Log::error('QR generation failed on update', ['point_id'=>$point->id, 'err'=>$e->getMessage()]);
            }
        }

        $point->save();

        return back()->with('ok', __('Waypoint updated.'));
    }






    public function updateBySerial(Request $request, $serial)
    {
        // Read from route to be 100% sure
        $routeParams = $request->route()?->parameters() ?? [];
        $serialFromRoute = $routeParams['serial'] ?? $serial ?? '';

        $serial = trim((string)$serialFromRoute);

        \Log::info('updateBySerial: incoming', [
            'route_params' => $routeParams,
            'serial'       => $serial,
            'url'          => $request->fullUrl(),
        ]);

        // Reject obvious locale-collisions
        if (in_array(strtolower($serial), ['fr','en'], true)) {
            \Log::warning('updateBySerial: serial looks like locale', ['serial' => $serial]);
            return back()->withErrors(['serial' => __('Invalid token provided.')])->withInput();
        }

        // Optional: whitelist tokens (base64/url-safe or your VER-… pattern)
        if (!preg_match('/^[A-Za-z0-9\-\_=]+$/', $serial)) {
            \Log::warning('updateBySerial: bad format', ['serial' => $serial]);
            return back()->withErrors(['serial' => __('Invalid token format.')])->withInput();
        }

        $p = \App\Models\Point::where('serial', $serial)->first();

        if (!$p) {
            \Log::warning('updateBySerial: not found', ['serial' => $serial]);
            return back()->withErrors(['serial' => __('Waypoint not found for this token.')]);
        }

        $data = $request->validate([
            'name'      => ['required','string','max:255'],
            'latitude'  => ['required','numeric','between:-90,90'],
            'longitude' => ['required','numeric','between:-180,180'],
            'altitude'  => ['nullable','numeric'],
            'floor'     => ['nullable','integer'],
            'favoris'   => ['nullable','integer','in:0,1'],
            'address1'  => ['nullable','string','max:255'],
            'address2'  => ['nullable','string','max:255'],
            'postcode'  => ['nullable','string','max:32'],
            'city'      => ['nullable','string','max:255'],
            'country_isocode' => ['nullable','string','max:8'],
            'comment'   => ['nullable','string'],
        ]);

        $p->update($data);

        return back()->with('status', __('Waypoint updated.'));
    }




  




    public function updatePhotoBySerial(Request $request)
    {
        $data = $request->validate([
            'serial' => ['required','string','exists:points,serial'],
            'photo'  => ['required','image','max:4096'],
        ]);

        $point = \App\Models\Point::where('serial', $data['serial'])->firstOrFail();

        $path = $request->file('photo')->store('points/photos', 'public');
        $point->update(['photo' => $path]);

        return back()->with('ok', __('Photo updated.'));
    }






    /** Protected delete */
    public function destroy(Point $point)
    {
        if (auth()->id() !== 1 && $point->user_id !== auth()->id()) {
            abort(403);
        }

        $point->delete();
        return back()->with('ok','Point supprimé.');
    }






    /** Reverse geocoding → normalized array */
    private function reverseGeocodeArr($lat, $lng): array
    {
        try {
            $ua    = config('app.name','Laravel').'/'.app()->version();
            $email = config('mail.from.address');

            $res = Http::withHeaders(['User-Agent' => $ua])
                ->timeout(8)
                ->get('https://nominatim.openstreetmap.org/reverse', [
                    'format'         => 'jsonv2',
                    'lat'            => $lat,
                    'lon'            => $lng,
                    'zoom'           => 18,
                    'addressdetails' => 1,
                    'email'          => $email,
                ]);

            if (!$res->ok()) return [];

            $j = $res->json();
            $a = $j['address'] ?? [];

            $address1 = trim(($a['house_number'] ?? '').' '.($a['road'] ?? ''));
            $address2 = $a['suburb'] ?? ($a['neighbourhood'] ?? null);
            $city     = $a['city'] ?? $a['town'] ?? $a['village'] ?? $a['hamlet'] ?? null;
            $postcode = $a['postcode'] ?? null;
            $cc       = strtoupper($a['country_code'] ?? '');

            return [
                'address1'        => $address1 ?: null,
                'address2'        => $address2,
                'postcode'        => $postcode,
                'city'            => $city,
                'country_isocode' => $cc ?: null,
            ];
        } catch (\Throwable $e) {
            return [];
        }
    }


    



 

    public function toggleFavorite(Request $request, string $locale, string $token)
    {

        $user = $request->user();
        if (!$user) {
            return response()->json(['error' => 'unauthenticated'], 401);
        }

        $point = Point::where('serial', $token)->first();
        if (!$point) {
            return response()->json(['error' => 'not_found'], 404);
        }

        $exists = $user->favoritePoints()
                    ->where('points.id', $point->id)
                    ->exists();

        if ($exists) {
            $user->favoritePoints()->detach($point->id);
            $favoris = 0;
        } else {
            $user->favoritePoints()->attach($point->id);
            $favoris = 1;
        }

        return response()->json(['favoris' => $favoris]);


     }







    /** Search */

    public function search(Request $request)
    {
        // 1) Guests → redirect to the localized login page
        if (!auth()->check()) {
            return redirect(lr('login'));
        }
    
        $user   = auth()->user();
        $userId = $user->id;
    
        $base = Point::with('owner:id,firstname,lastname,email')
            ->visibleFor($user) // safe to call because we already have an authenticated user
            ->latest();

        // Default values passed to the view
        $societies          = [];
        $selectedSocietyId  = null;
        $usersOfSociety     = [];
    
        if ($userId === 1) {
            if (class_exists(\App\Models\Society::class) && \Schema::hasTable('societies')) {
                $societies         = Society::orderBy('name')->get(['id','name']);
                $selectedSocietyId = (int) $request->query('society_id', 0) ?: null;
    
                if ($selectedSocietyId) {
                    $usersOfSociety = User::whereHas('societies', fn($q) =>
                        $q->where('society_id', $selectedSocietyId)
                    )->orderBy('lastname')->get(['id','firstname','lastname']);
    
                    $base->whereIn('user_id', $usersOfSociety->pluck('id'));
                }
            }
        }
    
        $points = $base->paginate(20);
    
        return view('points.search', [
            'points'            => $points,
            'societies'         => $societies,
            'selectedSocietyId' => $selectedSocietyId,
            'usersOfSociety'    => $usersOfSociety,
            'q'                 => '',
        ]);
    }






    public function result(Request $request)
    {
        // Delegate to search() to avoid duplicating logic
        return $this->search();
    }


  



 

    
   
    public function publicShow(string $locale, string $token)
    {
        $p = $this->resolvePointFromToken($token);

        if (!$p) {
            // Return a clean 404 view with a helpful message
            return response()->view('points.public_show', [
                'p'        => null,
                'readonly' => false,
                'reason'   => 'not_found',
                'serialToken' => $token, // useful for fallback links
            ], 404);
        }

        // Standard page (editable according to your auth rules in the view)
        return view('points.public_show', ['p' => $p, 'serialToken' => $token, 'readonly' => false]);
    }






    public function publicView(string $locale, string $token)
    {
        app()->setLocale($locale);

        try {
            $serial = trim($token);

           // echo $token; exit;

            \Log::info('publicView: params', [
                'route_params' => request()->route()->parameters(),
                'locale' => $locale,
                'token'  => $serial,
            ]);

            $p = \App\Models\Point::where('serial', $serial)->first();

            if (!$p) {
                \Log::warning('publicView: serial not found', ['token' => $serial]);
                return response()->view('points.public_show', [
                    'p'           => null,
                    'serialToken' => $serial,    // ← restera bien le token, pas 'fr'
                    'photoUrl'    => null,
                    'qrUrl'       => null,
                    'readonly'    => true,
                    'reason'      => 'not_found',
                ], 404);
            }

            $photoUrl = $p->photo ? \Storage::url($p->photo) : null;
            $qrUrl    = $p->qrcode ? \Storage::url($p->qrcode) : null;

            return view('points.public_show', [
                'p'           => $p,
                'serialToken' => $serial,
                'photoUrl'    => $photoUrl,
                'qrUrl'       => $qrUrl,
                'readonly'    => true,
            ]);

        } catch (\Throwable $e) {
            \Log::error('publicView: exception', ['token' => $serial, 'err' => $e->getMessage()]);
            return response()->view('points.public_show', [
                'p'           => null,
                'serialToken' => $serial,
                'photoUrl'    => null,
                'qrUrl'       => null,
                'readonly'    => true,
                'reason'      => 'exception',
            ], 404);
        }
    }







    /** Resolve a point from any token (serial, numeric id, base64 legacy). */
    private function resolvePointFromToken(?string $token): ?Point
    {
        $token = trim((string) $token);

        if ($token === '') {
            return null;
        }

        $point = Point::where('serial', $token)->first();
        if ($point) {
            return $point;
        }

        if (ctype_digit($token)) {
            $point = Point::find((int) $token);
            if ($point) {
                return $point;
            }
        }

        $decoded = base64_decode(strtr($token, '-_', '+/'), true);
        if ($decoded !== false && preg_match('/^(\d+)-/', $decoded, $matches)) {
            $decodedId = (int) $matches[1];
            if ($decodedId > 0) {
                $point = Point::find($decodedId);
                if ($point) {
                    return $point;
                }
            }
        }

        return Point::whereRaw('LOWER(serial) = ?', [strtolower($token)])->first();
    }








    private function isSuperAdmin($user): bool
    {
        // Adjust if you use a different rule (e.g. role_id == 1)
        return (int)($user->id) === 1 || (int)($user->role_id ?? 0) === 1;
    }








    public function share(string $locale, Request $request, string $token)
    {
        $owner = $request->user();
        abort_unless($owner, 401);

        //echo $token; exit;

        // Resolve model from serial first, then id
        $model = Point::where('serial', $token)->first();
        if (!$model && ctype_digit($token)) {
            $model = Point::find((int)$token);
        }
        abort_unless($model, 404, 'Point introuvable');

        // Authorization: owner or super admin
        $isSuper = (int)$owner->id === 1 || (int)($owner->role_id ?? 0) === 1;
        abort_unless($isSuper || (int)$model->user_id === (int)$owner->id, 403);

        // Validate email + optional window
        $data = $request->validate([
            'email'    => ['required','email'],
            'start_at' => ['nullable','date'],
            'end_at'   => ['nullable','date','after_or_equal:start_at'],
        ]);

        $recipientEmail = $data['email'];
        $recipientUser  = \App\Models\User::where('email', $recipientEmail)->first(); // can be null



        if ($recipientUser) {
            $sharing = Sharing::updateOrCreate(
                [
                    'owner_user_id'  => $owner->id,
                    'point_id'       => $model->id,
                    'target_user_id' => $recipientUser->id,
                ],
                [
                    'recipient_email' => $recipientEmail, // keep for display/history
                    'start_at'        => $data['start_at'] ?? null,
                    'end_at'          => $data['end_at']   ?? null,
                ]
            );
        } else {
            $sharing = Sharing::updateOrCreate(
                [
                    'owner_user_id'   => $owner->id,
                    'point_id'        => $model->id,
                    'recipient_email' => $recipientEmail,
                ],
                [
                    'target_user_id' => null,
                    'start_at'       => $data['start_at'] ?? null,
                    'end_at'         => $data['end_at']   ?? null,
                ]
            );
        }

        // Send the email (works for users and non-users)
        Mail::to($recipientEmail)->send(
            new PointSharedMail($model, $owner, $recipientEmail, $sharing, $recipientUser)
        );

        return back()->with('ok', __('Waypoint shared.'));
    }


    





    public function sharePoint(Request $request, Point $point)
    {
        $user = $request->user();
    
        // Authorization: point owner or super admin
        abort_unless($user && ($this->isSuperAdmin($user) || $user->id === $point->user_id), 403);
    
        $data = $request->validate([
            'target_user_id' => ['required','exists:users,id','different:owner_user_id'],
            'start_at'       => ['nullable','date'],
            'end_at'         => ['nullable','date','after_or_equal:start_at'],
        ]);
    
        // Prevent sharing with yourself
        if ((int)$data['target_user_id'] === (int)$user->id) {
            return back()->withErrors(['target_user_id' => "Impossible de se partager le waypoint à soi-même."]);
        }
    
        Sharing::create([
            'owner_user_id'  => $user->id,
            'point_id'       => $point->id,
            'target_user_id' => $data['target_user_id'],
            'start_at'       => $data['start_at'] ?? null,
            'end_at'         => $data['end_at'] ?? null,
        ]);
    
        return back()->with('ok', 'Partage créé avec succès.');
    }







}
