<?php
    // app/Http/Controllers/Pro/ProItineraryController.php
    namespace App\Http\Controllers\Pro;

    use App\Http\Controllers\Controller;
    use App\Models\Society;
    use App\Models\{User, Point, Itinerary, ItineraryPoint, Sharing};
    use Illuminate\Http\Request;
    use Illuminate\Support\Facades\Auth;
    use Illuminate\Validation\Rule;
    use Illuminate\Support\Facades\DB;
    use Illuminate\Support\Facades\Log;
    
   

    class ProItineraryController extends Controller
    {

        /** List itineraries for current scope (Pro admin sees his own; Admin sees all) */
        public function index()
        {
            $me = Auth::user();
            
            $query = \App\Models\Itinerary::query()
                ->with([
                    'owner:id,firstname,lastname,email',     // 👈 eager-load owner
                    'users:id,firstname,lastname,email',     // 👈 eager-load assignees
                ])
                ->latest('id');

            if ((int)$me->id !== 1) {
                $query->where(function($q) use ($me) {
                    $q->where('owner_user_id', $me->id)
                    ->orWhereHas('users', fn($s) => $s->where('users.id', $me->id));
                });
            }

            $itineraries = $query->paginate(20);

            return view('pro.itins.index', compact('itineraries'));
        }




        /** Show the pro admin create form */
        public function create(Request $request)
        {
            $me = $request->user();

            // Get the (first) society of this pro admin.
            // If multiple, you might add UI to choose; for now we take the first.
            $society = $me->societies()
                ->select('societydb.id', 'societydb.name') // 👈 fully-qualified
                ->first();

            // Company members (id > 1)
            $companyUsers = collect();
            if ($society) {
                $companyUsers = User::where('id', '>', 1)
                    ->whereHas('societies', fn($q) => $q->where('society_id', $society->id))
                    ->orderBy('lastname')->orderBy('firstname')
                    ->get(['id','firstname','lastname','email']);
            }

            return view('pro.itins.create', [
                'society'      => $society,
                'companyUsers' => $companyUsers,
            ]);
        }





        // JSON: waypoints visibles pour l’admin pro (courant)
        public function myPoints(Request $request)
        {
            $me = $request->user();

            Log::info('myPoints: incoming', ['user_id' => $me?->id]);

            try {
                // ->visibleFor($me) si tu l’as ; sinon fallback simple
                $points = Point::query()
                    // ->visibleFor($me)
                    ->where('user_id', $me->id)
                    ->orderBy('name')
                    ->get(['id','name','latitude','longitude']);

                Log::info('myPoints: result', ['count' => $points->count()]);
                return response()->json($points);
            } catch (\Throwable $e) {
                Log::error('myPoints: exception', ['err' => $e->getMessage()]);
                return response()->json(['error' => 'server_error', 'message' => $e->getMessage()], 500);
            }
        }
        





        // POST /pro/itineraries – attache aussi les points + sort_order
        public function store(Request $request)
        {
            $me = $request->user();
        
            $data = $request->validate([
                'name'               => ['required', 'string', 'max:255', 'unique:itineraries,name'],
                'society_id'         => ['required', 'exists:societydb,id'],
                'assigned_user_ids'  => ['nullable', 'array'],
                'assigned_user_ids.*'=> ['integer', 'exists:users,id'],
        
                'point_ids'          => ['nullable','array'],
                'point_ids.*'        => ['integer','exists:points,id'],
            ]);
        
        $it = \App\Models\Itinerary::create([
            'name'             => $data['name'],
            'owner_user_id'    => $me->id,
            'assigned_user_id' => $me->id,
            'society_id'       => $data['society_id'],
            'scheduled_date'   => now(),
        ]);

        $assignedUserIdsInput = $request->input('assigned_user_ids', []);
        if (!is_array($assignedUserIdsInput)) {
            $assignedUserIdsInput = array_filter([$assignedUserIdsInput]);
        }

        $assignedUserIds = collect($assignedUserIdsInput)
            ->filter(fn($v) => is_numeric($v))
            ->map(fn($v) => (int) $v)
            ->filter(fn($v) => $v > 0)
            ->unique()
            ->values()
            ->all();
        $it->users()->sync($assignedUserIds);
        
            $pointIds = array_values(array_unique($data['point_ids'] ?? []));
            if ($pointIds) {
                $attach = [];
                foreach ($pointIds as $i => $pid) {
                    $attach[$pid] = ['sort_order' => $i + 1];
                }
                $it->points()->syncWithoutDetaching($attach);
            }
        
            return redirect()->route('pro.itins.index', ['locale'=>app()->getLocale()])
                ->with('status', __('Itinerary successfully created.'));
        }






        public function show(Itinerary $itinerary) { return $this->edit($itinerary); }





        
        public function edit(\Illuminate\Http\Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);
            
          
            // Fetch itinerary with required relations for the view
            $it = Itinerary::query()
                ->with([
                    'owner:id,firstname,lastname,email,role_id',
                    'owner.societies:id,name',
                    'users:id,firstname,lastname,email',
                    'items.point:id,name,city,latitude,longitude',
                ])
                ->findOrFail((int) $itineraryId);

            if (!$it) {
                \Log::warning('ProItineraryController@edit: not found', ['id' => $itineraryId]);
                abort(404);
            }

            $me              = $request->user();
            $isAdmin         = (int)$me->id === 1;
            $isOwner         = (int)$it->owner_user_id === (int)$me->id;
            $isPivotAssignee = $it->users->contains(fn($u) => (int)$u->id === (int)$me->id);
            $isAssignedUser  = (int)$it->assigned_user_id === (int)$me->id;

            abort_unless($isAdmin || $isOwner || $isPivotAssignee || $isAssignedUser, 403);

            // --- Resolve target society (ALWAYS define before using it) ---
            $targetSocietyId = $this->resolveTargetSocietyId($it, $me);
            Log::info('edit: resolved target society', [
                'it_id' => $it->id,
                'society_id' => $targetSocietyId,
            ]);
    
            // Admin pros list (filter by society when known)
            $adminPros = collect();
            $adminPros = User::query()
                ->where('role_id', 2)
                ->when($targetSocietyId, fn($q) => $q->whereHas('societies', fn($s) => $s->where('society_id', $targetSocietyId)))
                ->with(['societies:id,name'])
                ->orderBy('lastname')
                ->orderBy('firstname')
                ->get(['id','firstname','lastname','email']);

            // Company users for checkbox list (members of target society, id > 1)
            $companyUsers = collect();
            if ($targetSocietyId) {
                $companyUsers = User::query()
                    ->where('id', '>', 1)
                    ->whereHas('societies', fn($q) => $q->where('society_id', $targetSocietyId))
                    ->orderBy('lastname')
                    ->orderBy('firstname')
                    ->get(['id','firstname','lastname','email']);
            }

            $assignedIds = $it->users->pluck('id')->all();

            return view('pro.itins.edit', [
                'itinerary'         => $it,
                'targetSocietyId'   => $targetSocietyId,
                'adminPros'         => $adminPros,
                'companyUsers'      => $companyUsers,
                'assignedIds'       => $assignedIds,
            ]);

        }





        public function update(\Illuminate\Http\Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);
 
            $me = auth()->user();
        
        // Resolve model (with relations if needed)
        $itinerary = Itinerary::query()
            ->with(['owner:id,role_id', 'items', 'users:id']) // items to update sort/orders and timestamps
            ->findOrFail((int)$itineraryId);

        // Authorization: admin or owner
        $isAdmin = (int)$me->id === 1;
        $isOwner = (int)$itinerary->owner_user_id === (int)$me->id;
        $isAssignee = $itinerary->users->contains(fn($u) => (int)$u->id === (int)$me->id);
        $isAssignedUser = (int)$itinerary->assigned_user_id === (int)$me->id;
        abort_unless($isAdmin || $isOwner || $isAssignee || $isAssignedUser, 403);
        
            // Validate top-level fields and per-item updates
            $data = $request->validate([
                'name'            => ['required','string','max:255'],
                'scheduled_date'  => ['nullable','date'],
                'assigned_user_ids'  => ['nullable','array'],
                'assigned_user_ids.*'=> ['integer','exists:users,id'],
                'items'                  => ['nullable','array'],
                'items.*.sort_order'     => ['nullable','integer','min:1'],
                'items.*.arrival_at'     => ['nullable','date'],
                'items.*.validated_at'   => ['nullable','date'],
                'items.*.rescheduled_at' => ['nullable','date'],
                'items.*.comment'        => ['nullable','string','max:2000'],
            ]);
        
            // Enforce name uniqueness per owner (if you want strict unique at update time)
            $exists = Itinerary::where('owner_user_id', $itinerary->owner_user_id)
                ->where('name', $data['name'])
                ->where('id','!=',$itinerary->id)
                ->exists();
            abort_if($exists, 422, 'Name must be unique for this organizer.');
        
            $assignedUserIdsInput = $request->input('assigned_user_ids', []);
            if (!is_array($assignedUserIdsInput)) {
                $assignedUserIdsInput = array_filter([$assignedUserIdsInput]);
            }

            $assignedUserIds = collect($assignedUserIdsInput)
                ->filter(fn($v) => is_numeric($v))
                ->map(fn($v) => (int) $v)
                ->filter(fn($v) => $v > 0)
                ->unique()
                ->values()
                ->all();

            DB::transaction(function () use ($itinerary, $data, $assignedUserIds) {
                // Update main fields
                $itinerary->name           = $data['name'];
                $itinerary->scheduled_date = $data['scheduled_date'] ?? null;
                $itinerary->save();

                $itinerary->users()->sync($assignedUserIds);
        
                // Update per-point data if provided
                foreach (($data['items'] ?? []) as $itemId => $row) {
                    /** @var ItineraryPoint|null $itp */
                    $itp = $itinerary->items->firstWhere('id', (int)$itemId);
                    if (!$itp) continue;
        
                    $itp->sort_order     = isset($row['sort_order'])     ? (int)$row['sort_order'] : $itp->sort_order;
                    $itp->arrival_at     = $row['arrival_at']     ?? $itp->arrival_at;
                    $itp->validated_at   = $row['validated_at']   ?? $itp->validated_at;
                    $itp->rescheduled_at = $row['rescheduled_at'] ?? $itp->rescheduled_at;
                    $itp->comment        = $row['comment']        ?? $itp->comment;
        
                    // keep boolean "done" in sync if you still use it for UI
                    $itp->done = !empty($row['validated_at']) ? 1 : 0;
        
                    $itp->save();
                }
            });
        
            // Redirect back to index (localized for pro)
            return redirect()->to(
                $isAdmin
                    ? route('admin.itins.index')
                    : route('pro.itins.index', ['locale'=>app()->getLocale()])
            )->with('ok', __('Itinerary updated'));
        }





        public function destroy(Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);
            $me = Auth::user();
            $this->authorizeAccess($me, $itinerary);

            $itinerary->delete();
            return redirect()->route($this->baseRoute().'.itins.index')->with('ok', __('Itinerary deleted'));
        }





        /** Attach points (add to end) */
        public function attachPoints(Request $request)
        {
            // Robust resolution even with /{locale}/pro/... prefix
            $raw = $request->route('itinerary');
            $itinerary = $raw instanceof Itinerary ? $raw : Itinerary::findOrFail((int)$raw);
        
            $me = auth()->user();
            $this->authorizeAccess($me, $itinerary);
        
            $data = $request->validate([
                'point_ids'   => ['required','array','min:1'],
                'point_ids.*' => ['integer','exists:points,id'],
            ]);
        
            // Avoid duplicates; keep increasing sort_order
            $existing = $itinerary->items()->pluck('point_id')->all();
            $toAdd   = collect($data['point_ids'])->map(fn($v)=>(int)$v)
                        ->reject(fn($pid) => in_array($pid, $existing))
                        ->values();
        
            if ($toAdd->isNotEmpty()) {
                $startOrder = (int) $itinerary->items()->max('sort_order');
                $order = $startOrder > 0 ? $startOrder + 1 : 1;
        
                \DB::transaction(function () use ($itinerary, $toAdd, &$order) {
                    foreach ($toAdd as $pid) {
                        ItineraryPoint::create([
                            'itinerary_id' => $itinerary->id,
                            'point_id'     => $pid,
                            'sort_order'   => $order++,
                            // other fields (done, timestamps…) default/null
                        ]);
                    }
                });
            }
        
            return back()->with('ok', __('Waypoints added to itinerary'));
        }





        /** Drag & drop reorder: receive array of item IDs in new order */
        public function reorder(Request $request, Itinerary $itinerary)
        {
            $me = Auth::user();
            $this->authorizeAccess($me, $itinerary);

            $ids = $request->validate([
                'item_ids' => ['required','array','min:1'],
                'item_ids.*' => ['integer','exists:itinerary_points,id'],
            ])['item_ids'];

            DB::transaction(function () use ($itinerary, $ids) {
                $order = 1;
                foreach ($ids as $id) {
                    ItineraryPoint::where('itinerary_id', $itinerary->id)
                        ->where('id', $id)
                        ->update(['sort_order' => $order++]);
                }
            });

            return response()->json(['ok' => true]);
        }




        /** Update a single itinerary item (done, planned_at, rescheduled_at, comment) */
        public function updateItem(Request $request, Itinerary $itinerary, ItineraryPoint $item)
        {
            $me = Auth::user();
            $this->authorizeAccess($me, $itinerary);
            abort_unless($item->itinerary_id === $itinerary->id, 404);

            $data = $request->validate([
                'planned_at'     => ['nullable','date'],
                'done'           => ['nullable','boolean'],
                'rescheduled_at' => ['nullable','date'],
                'comment'        => ['nullable','string','max:2000'],
            ]);

            $item->update([
                'planned_at'     => $data['planned_at'] ?? $item->planned_at,
                'done'           => array_key_exists('done', $data) ? (bool)$data['done'] : $item->done,
                'rescheduled_at' => $data['rescheduled_at'] ?? $item->rescheduled_at,
                'comment'        => $data['comment'] ?? $item->comment,
            ]);

            return response()->json(['ok' => true]);
        }




        public function removeItem(Itinerary $itinerary, ItineraryPoint $item)
        {
            $me = Auth::user();
            $this->authorizeAccess($me, $itinerary);
            abort_unless($item->itinerary_id === $itinerary->id, 404);

            $item->delete();
            return back()->with('ok', __('Waypoint removed from itinerary'));
        }




        // ---------- helpers ----------

        /** Points owned by current admin pro OR shared to him (date window honored) */
        protected function availablePointsFor(User $me)
        {
            // owned
            $owned = Point::query()->where('user_id', $me->id);

            // shared to me (via 'sharings' table)
            $now = now();
            $sharedPointIds = Sharing::query()
                ->where('target_user_id', $me->id)
                ->where(function($q) use ($now) {
                    $q->whereNull('start_at')->orWhere('start_at', '<=', $now);
                })
                ->where(function($q) use ($now) {
                    $q->whereNull('end_at')->orWhere('end_at', '>=', $now);
                })
                ->pluck('point_id');

            return Point::with('owner:id,firstname,lastname')
                ->where(function($q) use ($me, $sharedPointIds) {
                    $q->where('user_id', $me->id)
                    ->orWhereIn('id', $sharedPointIds);
                })
                ->orderBy('name')
                ->get(['id','name','city','serial','user_id']);
        }




        private function normalizeItineraryArgs($a, $b = null): array
        {
            // When called from Pro: ($a='fr', $b='2')
            // When called from Admin: ($a='2', $b=null)
            if (is_numeric($a)) {
                return [app()->getLocale(), (int)$a];
            }
            return [(string)$a, (int)$b];
        }


        protected function baseRoute(): string
        {
            return (int)Auth::id() === 1 ? 'admin' : 'pro';
        }




        /** Users Pro (role_id=4) assignables dans le périmètre */
        protected function eligibleAssignees(User $me)
        {
            if ((int)$me->id === 1) {
                // Admin : tous les users pro
                return User::where('role_id', 4)
                    ->orderBy('lastname')->orderBy('firstname')
                    ->get(['id','firstname','lastname','email']);
            }
        
            // Admin Pro : users pro liés à ses sociétés
            $societyIds = $me->societies()->pluck('societydb.id');
            if ($societyIds->isEmpty()) return collect();
        
            return User::where('role_id', 4)
                ->whereHas('societies', fn($q) => $q->whereIn('societydb.id', $societyIds))
                ->orderBy('lastname')->orderBy('firstname')
                ->get(['id','firstname','lastname','email']);
        }




        /** Users Pro (role_id=4) rattachés au périmètre de l’owner (organisateur) */
        protected function eligibleAssigneesForOwner(\App\Models\User $owner)
        {
            if ((int)$owner->id === 1) {
                // improbable ici, mais on garde un fallback
                return \App\Models\User::where('role_id', 4)
                    ->orderBy('lastname')->orderBy('firstname')
                    ->get(['id','firstname','lastname','email']);
            }

            // Users pro liés aux mêmes sociétés que l’owner (admin pro)
            $societyIds = $owner->societies()->pluck('societydb.id');
            if ($societyIds->isEmpty()) return collect();

            return \App\Models\User::where('role_id', 4)
                ->whereHas('societies', fn($q) => $q->whereIn('societydb.id', $societyIds))
                ->orderBy('lastname')->orderBy('firstname')
                ->get(['id','firstname','lastname','email']);
        }




        /** Owner’s own points + points shared to owner (active window) */
        protected function availablePointsForOwner(User $owner)
        {
            $now = now();
            $sharedPointIds = Sharing::query()
                ->where('target_user_id', $owner->id)
                ->where(function($q) use ($now) { $q->whereNull('start_at')->orWhere('start_at','<=',$now); })
                ->where(function($q) use ($now) { $q->whereNull('end_at')->orWhere('end_at','>=',$now); })
                ->pluck('point_id');

            return Point::with('owner:id,firstname,lastname')
                ->where(function($q) use ($owner, $sharedPointIds) {
                    $q->where('user_id', $owner->id)->orWhereIn('id', $sharedPointIds);
                })
                ->orderBy('name')
                ->get(['id','name','city','latitude','longitude','serial','user_id']);
        }




        /**
         * Resolve the target society id for an itinerary and viewer.
         * Priority:
         *  1) itinerary.society_id
         *  2) first society of itinerary owner
         *  3) first society of viewer (when viewer is not global admin)
         */
        private function resolveTargetSocietyId(Itinerary $it, User $viewer): ?int
        {
            if ($it->society_id) {
                return (int) $it->society_id;
            }
            $ownerSoc = $it->owner?->societies?->first()?->id ?? null;
            if ($ownerSoc) {
                return (int) $ownerSoc;
            }
            if ((int) $viewer->id !== 1) {
                return optional($viewer->societies()->first())->id ? (int) $viewer->societies()->first()->id : null;
            }
            return null;
        }





        // Accept both shapes: (Request, $itineraryId) for Admin and (Request, $locale, $itineraryId) for Pro
        public function map(\Illuminate\Http\Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);

            \Log::info('ProItineraryController@map: incoming', [
                'user_id' => auth()->id(),
                'locale'  => $locale,
                'itinerary_id' => $itineraryId,
            ]);

            $me = $request->user();

            /** @var \App\Models\Itinerary|null $it */
            $it = \App\Models\Itinerary::with([
                'items.point:id,name,city,latitude,longitude',
                'users:id,firstname,lastname',
                'owner:id,firstname,lastname',
            ])->find($itineraryId);

            if (!$it) {
                abort(404);
            }

            $isAdmin    = (int)$me->id === 1;
            $isOwner    = (int)$it->owner_user_id === (int)$me->id;
            $isAssignee = $it->users->contains(fn($u) => (int)$u->id === (int)$me->id);
            abort_unless($isAdmin || $isOwner || $isAssignee, 403);

            return view('pro.itins.map', [
                'itinerary' => $it,
            ]);
        }
        




        public function saveProgress(Request $request, $a, $b = null)
        {
            // Normalize args for Admin (no locale) vs Pro (with locale)
            // When Pro: $a='fr', $b='2' ; When Admin: $a='2', $b=null
            $isNumericFirst = is_numeric($a);
            $locale        = $isNumericFirst ? app()->getLocale() : (string)$a;
            $itineraryId   = (int) ($isNumericFirst ? $a : $b);
        
            \Log::info('ProItineraryController@saveProgress: incoming', [
                'user_id' => auth()->id(),
                'locale'  => $locale,
                'itinerary_id' => $itineraryId,
            ]);
        
            $me = $request->user();
        
            /** @var Itinerary|null $it */
            $it = Itinerary::query()
                ->with(['users:id', 'items']) // items needed to update done/timestamps
                ->find($itineraryId);
        
            if (!$it) {
                \Log::warning('saveProgress: not found', ['id'=>$itineraryId]);
                abort(404);
            }
        
            // Authorization: admin (id=1), owner or assigned user
            $isAdmin    = (int)$me->id === 1;
            $isOwner    = (int)$it->owner_user_id === (int)$me->id;
            $isAssignee = $it->users->contains(fn($u) => (int)$u->id === (int)$me->id);
            abort_unless($isAdmin || $isOwner || $isAssignee, 403);
        
            // Validate payload
            $data = $request->validate([
                // items[<itinerary_point_id>][done]            => 'on' (checkbox)
                // items[<id>][validated_at] / [rescheduled_at] => datetime-local
                // items[<id>][arrival_at], items[<id>][comment]
                'items'                        => ['required','array'],
                'items.*.done'                 => ['nullable','in:on,1,0'],
                'items.*.arrival_at'           => ['nullable','date'],
                'items.*.validated_at'         => ['nullable','date'],
                'items.*.rescheduled_at'       => ['nullable','date'],
                'items.*.comment'              => ['nullable','string','max:2000'],
            ]);
        
            DB::transaction(function () use ($it, $data) {
                foreach ($data['items'] as $itpId => $row) {
                    /** @var ItineraryPoint|null $itp */
                    $itp = $it->items->firstWhere('id', (int)$itpId);
                    if (!$itp) continue;
        
                    // done checkbox logic
                    $isDone = isset($row['done']) && (string)$row['done'] !== '0';
                    $itp->done = $isDone ? 1 : 0;
        
                    // timestamps (UI may send them or we can auto-fill validated_at when done)
                    $itp->arrival_at     = $row['arrival_at']     ?? $itp->arrival_at;
                    $itp->validated_at   = $row['validated_at']   ?? ($isDone ? now() : $itp->validated_at);
                    $itp->rescheduled_at = $row['rescheduled_at'] ?? $itp->rescheduled_at;
        
                    // optional comment
                    $itp->comment = $row['comment'] ?? $itp->comment;
        
                    $itp->save();
                }
        
                // If you track itinerary-level validation (e.g. validated_date), set it here if all done:
                // if ($it->items->every(fn($x) => (int)$x->done === 1)) {
                //     $it->validated_date = now();
                //     $it->save();
                // }
            });
        
            // Redirect back to the edit page (localized for pro)
            return redirect()->to(
                $isAdmin
                    ? route('admin.itins.edit', ['itinerary'=>$it->id])
                    : route('pro.itins.edit', ['locale'=>app()->getLocale(), 'itinerary'=>$it->id])
            )->with('ok', __('Progress saved.'));
        }
        



        public function validateItinerary(\App\Models\Itinerary $itinerary)
        {
            $me = \Auth::user();
            $this->authorizeAccess($me, $itinerary);
        
            $itinerary->update(['validated_date' => now()]);
            return back()->with('ok', __('Itinerary validated'));
        }




        public function attachUsers(\Illuminate\Http\Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);
            $me = auth()->user();
            $raw = $request->route('itinerary');
            $itinerary = $raw instanceof \App\Models\Itinerary ? $raw : \App\Models\Itinerary::findOrFail((int)$raw);

            // AuthZ: admin or owner/admin-pro who owns this itinerary
            $this->authorizeAccess($me, $itinerary);

            $data = $request->validate([
                'user_ids'   => ['required','array','min:1'],
                'user_ids.*' => ['integer','exists:users,id'],
            ]);

            // Limit attachable users: for admin → any pro (role_id=4),
            // for admin pro → only pros within owner's companies
            $allowed = $this->eligibleAssigneesForOwner($itinerary->owner)->pluck('id')->all();
            $toAttach = collect($data['user_ids'])->map(fn($id)=>(int)$id)->filter(fn($id)=>in_array($id, $allowed,true))->all();

            if (!empty($toAttach)) {
                $itinerary->users()->syncWithoutDetaching($toAttach);
            }

            return back()->with('ok', __('Users attached'));
        }




        public function detachUser(\Illuminate\Http\Request $request, $a, $b = null)
        {
            [$locale, $itineraryId] = $this->normalizeItineraryArgs($a, $b);
            $me = auth()->user();
            $rawIt = $request->route('itinerary');
            $rawUs = $request->route('user');

            $itinerary = $rawIt instanceof \App\Models\Itinerary ? $rawIt : \App\Models\Itinerary::findOrFail((int)$rawIt);
            $userId    = (int) $rawUs;

            $this->authorizeAccess($me, $itinerary);
            $itinerary->users()->detach($userId);

            return back()->with('ok', __('User detached'));
        }



        protected function authorizeAccess($me, \App\Models\Itinerary $itinerary): void
        {
            $isAdmin = (int)$me->id === 1;
            $isOwner = (int)$itinerary->owner_user_id === (int)$me->id;
            $isAssignedUser = (int)$itinerary->assigned_user_id === (int)$me->id;
            $hasLoadedUsers = $itinerary->relationLoaded('users');
            $isPivotAssignee = $hasLoadedUsers
                ? $itinerary->users->contains(fn($u) => (int)$u->id === (int)$me->id)
                : $itinerary->users()->where('users.id', $me->id)->exists();

            abort_unless($isAdmin || $isOwner || $isAssignedUser || $isPivotAssignee, 403);
        }


    }
