<?php
    // app/Http/Controllers/Pro/ProSocietyController.php
    namespace App\Http\Controllers\Pro;

    use App\Http\Controllers\Controller;
    use Illuminate\Http\Request;
    use App\Models\Society;
    use App\Support\ProSociety;
    use App\Models\User;

    class ProSocietyController extends Controller
    {

        /** Show edit form for the current Pro admin's company */
        public function edit(Request $request, string $locale)
        {
            $me    = $request->user();
            $socId = ProSociety::currentSocietyId($me);
            abort_if(!$socId, 403);
    
            $society = Society::findOrFail($socId);
    
            return view('pro.society.edit', [
                'society' => $society,
            ]);
        }



        public function show(Request $request, string $locale)
        {
            $me    = $request->user();
            $socId = ProSociety::currentSocietyId($me);
            abort_if(!$socId, 403);

            // Load society + ALL users (no role filter)
            $society = Society::with([
                'users' => fn($q) => $q->orderBy('lastname')->orderBy('firstname'),
            ])->findOrFail($socId);

            // If you prefer pagination for users list:
            $allUsers = User::whereHas('societies', fn($q) => $q->where('societydb.id', $socId))
                ->orderBy('lastname')->orderBy('firstname')
                ->paginate(50);

            return view('pro.society.show', [
                'society'  => $society,
                'allUsers' => $allUsers, // use this in the Blade to list everyone
            ]);
        }




        public function index(\Illuminate\Http\Request $request, string $locale)
        {
            $me = $request->user();
            $socId = \App\Support\ProSociety::currentSocietyId($me);
            abort_if(!$socId, 403);

            // IDs of company users (roles 2 and 4)
            $companyUserIds = \App\Models\User::query()
                ->whereHas('societies', fn($q) => $q->where('society_id', $socId))
                ->pluck('id');

            $itins = \App\Models\Itinerary::query()
                ->with(['owner:id,firstname,lastname','users:id,firstname,lastname'])
                ->where(function($q) use ($companyUserIds, $socId) {
                    $q->whereIn('owner_user_id', $companyUserIds)
                    ->orWhereHas('users', fn($qq) => $qq->whereIn('users.id', $companyUserIds))
                    ->orWhere('society_id', $socId);
                })
                ->orderByDesc('created_at')
                ->paginate(20);

            return view('pro.itins.index', ['itineraries' => $itins]);
        }



        /** Update company details (only within the admin pro company) */
        public function update(\Illuminate\Http\Request $request, string $locale)
        {
            $me    = $request->user();
            $socId = \App\Support\ProSociety::currentSocietyId($me);
            abort_if(!$socId, 403);

            $society = \App\Models\Society::findOrFail($socId);

            $data = $request->validate([
                'name'          => ['required','string','max:255'],
                'street'        => ['nullable','string','max:255'],
                'postcode'      => ['nullable','string','max:50'],
                'city'          => ['nullable','string','max:120'],
                'country'       => ['nullable','string','max:120'],
                'phone'         => ['nullable','string','max:50'],
                'email'         => ['nullable','email','max:255'],
                'subscribed_at' => ['nullable','date'],
                'photo'         => ['nullable','image','max:4096'], // 4MB
                'remove_photo'  => ['nullable','boolean'],
            ]);

            // Handle remove photo checkbox
            if (!empty($data['remove_photo'])) {
                if ($society->photo) {
                    \Storage::disk('public')->delete($society->photo);
                }
                $data['photo'] = null;
            } else {
                // Handle photo upload
                if ($request->hasFile('photo')) {
                    // optionally delete old file
                    if ($society->photo) {
                        \Storage::disk('public')->delete($society->photo);
                    }
                    $data['photo'] = $request->file('photo')->store('societies/photos', 'public');
                } else {
                    unset($data['photo']); // do not overwrite with null
                }
            }

            $society->fill($data)->save();

            return redirect()
                ->route('pro.society.edit', ['locale'=>$locale])
                ->with('ok', __('Company updated.'));
        }


    }