<?php

// app/Http/Controllers/Pro/ProUserController.php
namespace App\Http\Controllers\Pro;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Support\ProSociety;
use Illuminate\Validation\Rule;


class ProUserController extends Controller
{




    /** List pro users for the admin pro's society */
    public function index(Request $request, string $locale)
    {
        $me = $request->user();
        $socId = ProSociety::currentSocietyId($me);
        abort_if(!$socId, 403);

        // If you only want Admin Pro (2) + Pro (4), uncomment the whereIn:
        // ->whereIn('role_id', [2,4])
        $users = User::query()
            ->with(['societies:id,name']) // eager load for display
            ->whereHas('societies', fn($q) => $q->where('societydb.id', $socId))
            ->orderBy('lastname')
            ->orderBy('firstname')
            ->paginate(50);

        return view('pro.users.index', compact('users'));
    }




    public function create()
    {
        return view('pro.users.form', ['user' => new User()]);
    }




    public function store(Request $request)
    {
        $socId = (int)$request->attributes->get('pro_society_id');

        $data = $request->validate([
            'firstname' => ['required','string','max:100'],
            'lastname'  => ['required','string','max:100'],
            'email'     => ['required','email','max:255','unique:users,email'],
            'password'  => ['nullable','string','min:8'], // optional, we can generate
        ]);

        $pwd = $data['password'] ?? str()->random(12);

        DB::transaction(function () use ($data, $pwd, $socId) {
            $u = new User();
            $u->firstname = $data['firstname'];
            $u->lastname  = $data['lastname'];
            $u->email     = strtolower($data['email']);
            $u->password  = Hash::make($pwd);
            $u->role_id   = 4; // pro
            $u->blocked   = null;
            $u->save();

            // attach to the admin pro's society
            $u->societies()->syncWithoutDetaching([$socId]);
        });

        return redirect()->route('pro.users.index')->with('ok', __('User created.'));
    }




  
    /** Edit form for a user in the same company */
    public function edit(Request $request, string $locale, User $user)
    {
        $me = $request->user();
        $socId = ProSociety::currentSocietyId($me);
        abort_if(!$socId, 403);

        // target user must belong to the same society
        $inCompany = $user->societies()->where('societydb.id', $socId)->exists();
        abort_unless($inCompany, 403);

        // (optional) restrict which roles can be edited by a Pro Admin:
        // abort_unless(in_array((int)$user->role_id, [2,4]), 403);

        return view('pro.users.edit', [
            'user'    => $user,
            'society' => $me->societies()->where('societydb.id',$socId)->first(),
        ]);
    }



    /** Update user (same company) */
    public function update(Request $request, string $locale, User $user)
    {
        $me = $request->user();
        $socId = ProSociety::currentSocietyId($me);
        abort_if(!$socId, 403);

        $inCompany = $user->societies()->where('societydb.id', $socId)->exists();
        abort_unless($inCompany, 403);

        $data = $request->validate([
            'firstname' => ['nullable','string','max:100'],
            'lastname'  => ['nullable','string','max:100'],
            'email'     => ['required','email','max:255', Rule::unique('users','email')->ignore($user->id)],
            'phone'     => ['nullable','string','max:50'],
            'address1'  => ['nullable','string','max:255'],
            'address2'  => ['nullable','string','max:255'],
            'postcode'  => ['nullable','string','max:50'],
            'city'      => ['nullable','string','max:100'],
            'country_isocode' => ['nullable','string','max:10'],
            'password'  => ['nullable','string','min:8','confirmed'],
            'photo'     => ['nullable','image','max:2048'],
        ]);

        // Handle password change
        if (!empty($data['password'])) {
            $user->password = Hash::make($data['password']);
        }
        unset($data['password']);

        // Handle photo upload
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('photos', 'public');
            $user->photo = $path;
        }

        $user->fill($data)->save();

        return redirect()
            ->route('pro.users.index', ['locale' => $locale])
            ->with('ok', __('User updated successfully.'));
    }




    public function destroy(Request $request, User $user)
    {
        $socId = (int)$request->attributes->get('pro_society_id');

        if ((int)$user->role_id !== 4 || !$user->societies()->where('society_id', $socId)->exists()) {
            abort(403);
        }

        DB::transaction(function () use ($user, $socId) {
            // Detach from this society
            $user->societies()->detach($socId);

            // If user has no more societies, you can delete or leave as orphan.
            $stillLinked = $user->societies()->exists();
            if (!$stillLinked) {
                $user->delete();
            }
        });

        return back()->with('ok', __('User deleted.'));
    }




    /** Show CSV import form */
    public function importForm()
    {
        return view('pro.users.import');
    }




    /**
     * Accept a CSV file with headers:
     * firstname,lastname,email,password (password optional)
     * Upsert behavior:
     *  - existing by email => update names/role & attach society
     *  - new => create, set role_id=4, attach society
     */
    public function importCsv(Request $request)
    {
        $socId = (int)$request->attributes->get('pro_society_id');

        $data = $request->validate([
            'file' => ['required','file','mimes:csv,txt','max:10240'],
        ]);

        $path = $request->file('file')->getRealPath();
        $h = fopen($path, 'r');
        if (!$h) return back()->withErrors(['file' => 'Cannot open CSV file.']);

        $header = null;
        $rowNum = 0;
        $created = $updated = 0;
        DB::beginTransaction();
        try {
            while (($row = fgetcsv($h, 0, ',')) !== false) {
                $rowNum++;
                if ($rowNum === 1) {
                    $header = array_map(fn($v) => strtolower(trim($v)), $row);
                    continue;
                }
                $line = array_combine($header, array_map('trim', $row));

                $email = strtolower($line['email'] ?? '');
                if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
                    continue; // skip invalid
                }

                $firstname = $line['firstname'] ?? '';
                $lastname  = $line['lastname']  ?? '';
                $password  = $line['password']  ?? '';

                $user = User::whereRaw('LOWER(email)=?', [$email])->first();

                if ($user) {
                    // Only update pro inside this scope
                    if ((int)$user->role_id !== 4) {
                        // optionally: skip non-pro or force set to pro
                        $user->role_id = 4;
                    }
                    if ($firstname !== '') $user->firstname = $firstname;
                    if ($lastname  !== '') $user->lastname  = $lastname;
                    if ($password  !== '') $user->password  = Hash::make($password);
                    $user->save();

                    $user->societies()->syncWithoutDetaching([$socId]);
                    $updated++;
                } else {
                    $u = new User();
                    $u->firstname = $firstname ?: '';
                    $u->lastname  = $lastname  ?: '';
                    $u->email     = $email;
                    $u->password  = Hash::make($password !== '' ? $password : str()->random(12));
                    $u->role_id   = 4;
                    $u->blocked   = null;
                    $u->save();
                    $u->societies()->syncWithoutDetaching([$socId]);
                    $created++;
                }
            }
            fclose($h);
            DB::commit();
        } catch (\Throwable $e) {
            if (is_resource($h)) fclose($h);
            DB::rollBack();
            return back()->withErrors(['file' => 'Import failed: '.$e->getMessage()]);
        }

        return redirect()->route('pro.users.index')
            ->with('ok', __("Import done: :created created, :updated updated.", ['created'=>$created,'updated'=>$updated]));
    }
}