<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Mail;
use App\Mail\EmailChangeCodeMail;
use App\Services\SharingLinker;



class UserController extends Controller
{
 

    // GET /{locale}/user/edit
    public function edit(Request $request)
    {
        $user = $request->user();
        abort_unless($user, 403);

        $users = null;
        if ((int) $user->id === 1) {
            $users = User::with(['lastConnection'])->orderBy('name')->get();
        }

        return view('user.edit', [
            'user' => $user,
            'users' => $users,
        ]);
    }







    public function update(Request $request, $user)
    {
        \Log::debug('user.update:param', ['param' => $user]);

        $auth = $request->user();
        abort_unless($auth, 403);

        if ($auth->isAdmin()) {
            if (!$user instanceof User) {
                if (!ctype_digit((string) $user)) {
                    abort(404);
                }

                $user = User::findOrFail((int) $user);
            }
        } else {
            $user = $auth;
        }

        if ($auth->id !== $user->id && !$auth->isAdmin()) {
            abort(403);
        }

        $rules = [
            'firstname'        => ['nullable','string','max:255'],
            'lastname'         => ['nullable','string','max:255'],
            'phone'            => ['required','string','max:255'],
            'address1'         => ['nullable','string','max:255'],
            'address2'         => ['nullable','string','max:255'],
            'postcode'         => ['nullable','string','max:255'],
            'city'             => ['nullable','string','max:255'],
            'country_isocode'  => ['nullable','string','max:3'],
            'photo'            => ['nullable','image','max:5120'],
            'password'         => ['nullable','string','min:8','confirmed'],
        ];

        if ($auth->isAdmin()) {
            $rules['role_id']       = ['required','integer','exists:roledb,id'];
            $rules['society_ids']   = ['nullable','array'];
            $rules['society_ids.*'] = ['integer','exists:societydb,id'];
        }

        $data = $request->validate($rules);

        \Log::debug('user.update:start', [
            'auth_id' => $auth->id,
            'user_id' => $user->id,
            'fields'  => collect($data)->except(['password','password_confirmation','photo'])->all(),
        ]);

        if ($request->hasFile('photo')) {
            if ($user->photo && Storage::disk('public')->exists($user->photo)) {
                Storage::disk('public')->delete($user->photo);
            }

            $user->photo = $request->file('photo')->store('avatars', 'public');
        }

        if (!empty($data['password'])) {
            $user->password = Hash::make($data['password']);
        }

        foreach (['firstname','lastname','phone','address1','address2','postcode','city','country_isocode'] as $attribute) {
            if (array_key_exists($attribute, $data)) {
                $user->{$attribute} = $data[$attribute] === '' ? null : $data[$attribute];
            }
        }

        if ($auth->isAdmin() && array_key_exists('role_id', $data)) {
            $user->role_id = (int) $data['role_id'];
        }

        $changedKeys = array_keys($user->getDirty());
        $user->save();

        $societyIds = null;

        if ($auth->isAdmin()) {
            $societyIds = collect($data['society_ids'] ?? [])
                ->filter(fn($id) => filled($id))
                ->map(fn($id) => (int) $id)
                ->unique()
                ->values()
                ->all();

            $user->societies()->sync($societyIds);
        }

        \Log::debug('user.update:done', [
            'user_id' => $user->id,
            'changed' => $changedKeys,
            'societies' => $societyIds,
        ]);

        return redirect()
            ->route('user.edit', ['locale' => app()->getLocale()])
            ->with('ok', __('Profile updated.'));
    }





    // DELETE /{locale}/user/{user}
    
    public function destroy(string $locale, Request $request, User $user)
    {
        // Authorization as you need (example):
        // $this->authorize('delete', $user);

        $user->delete();

        // Optionally, add a flash message / redirect
        return redirect()->route('admin.users.index', ['locale' => $locale])
            ->with('ok', __('User deleted.'));
    }






    public function requestEmailChange(Request $request)
    {
        $user = Auth::user() ?? abort(403);

        $request->validate(['new_email' => ['required','email','unique:users,email']]);

        $code = (string) random_int(100000, 999999);
        Cache::put("email_change:{$user->id}", ['new'=>$request->new_email,'code'=>$code], now()->addMinutes(15));

        Mail::raw("Your Vereego verification code: {$code}", function ($m) use ($request) {
            $m->to($request->new_email)->subject('Verify your new email');
        });

        return back()->with('ok', __('A verification code has been sent to the new email.'));
    }







    public function requestEmailChangeCode(Request $request)
    {
        $user = $request->user();
    
        $v = $request->validate([
            //'new_email' => 'required|email|different:current_email',
            //'new_email' => 'required|email',
            'email' => 'required|email',
            //'current_email' => 'required|email',
        ]);
    
        // Basic security guard
        if (strcasecmp($v['current_email'], $user->email) !== 0) {
            return back()->withErrors(['current_email' => __('Current email does not match your account.')]);
        }

        $code = (string) random_int(100000, 999999);

        // Store the payload in cache (30 minutes)
        $key = "email_change:{$user->id}";
        Cache::put($key, ['code' => $code, 'new' => $v['new_email']], now()->addMinutes(30));
    
        Mail::to($v['new_email'])->send(new EmailChangeCodeMail($code, $v['new_email']));
    
        return back()->with('ok', __('A verification code has been sent to your email.'));
    }

    
    


    public function confirmEmailChange(Request $request, SharingLinker $linker)
    {
        $user = $request->user();
    
        $v = $request->validate(['code' => 'required|string']);
    
        $key = "email_change:{$user->id}";
        $payload = Cache::get($key);
    
        if (!$payload || $payload['code'] !== $v['code']) {
            return back()->withErrors(['code' => __('Invalid or expired code.')]);
        }
    
        // Update the email and mark it as verified
        $user->forceFill([
            'email' => $payload['new'],
            'email_verified_at' => now(),
        ])->save();
    
        $user = $request->user(); // may be null if not authenticated
        app(\App\Services\SharingLinker::class)->linkForUser($user); // now safe

        Cache::forget($key);
    
        return back()->with('ok', __('Your email has been updated and verified.'));
    }





    public function verifyEmail(Request $request, User $user)
    {
        $request->validate([
            'code' => ['required','digits:6'],
        ]);

        if ($user->email_verification_code === $request->code) {
            $user->email = $user->pending_email;
            $user->pending_email = null;
            $user->email_verification_code = null;
            $user->email_verified_at = now();
            $user->save();

            return back()->with('ok', 'Email confirmé avec succès !');
        }

        return back()->withErrors(['code' => 'Code invalide.']);
    }














}
