<?php

    namespace App\Http\Controllers;

    use Illuminate\Http\Request;
    use Illuminate\Validation\Rule;
    use Illuminate\Support\Facades\Hash;
    use Illuminate\Support\Facades\Storage;
    use Illuminate\Support\Facades\Log;

    class UserProfileController extends Controller
    {


        public function edit(Request $request)
        {
            $user = $request->user();

            $societies      = collect();
            $societyNames   = [];
            if ($user && (int)$user->id !== 1) {
                // relation Eloquent (nécessite Society::$table = 'societydb')
                try {
                    $user->loadMissing(['societies:id,name']);
                    $societies    = $user->societies;
                    $societyNames = $societies->pluck('name')->filter()->values()->all();
                } catch (\Throwable $e) {
                    // Fallback SQL si la relation n’est pas OK
                    $rows = \DB::table('societydb_user')
                        ->join('societydb', 'societydb.id', '=', 'societydb_user.society_id')
                        ->where('societydb_user.user_id', $user->id)
                        ->orderBy('societydb.name')
                        ->get(['societydb.id', 'societydb.name']);
                    $societies    = collect($rows);
                    $societyNames = $societies->pluck('name')->filter()->values()->all();
                }
            }

            Log::info('profile.edit hit', ['uid' => $user?->id, 'companies' => $companyNames]);
            

            return view('account.edit', [
                'user'          => $user,
                'societies'     => $societies,     // collection (pour plus tard)
                'societyNames'  => $societyNames,  // tableau simple → affichage garanti
            ]);
        }




        public function update(Request $request)
        {
            $user = $request->user();

            $data = $request->validate([
                'firstname'    => ['nullable','string','max:120'],
                'lastname'     => ['nullable','string','max:120'],
                'email'        => ['required','email','max:190', Rule::unique('users','email')->ignore($user->id)],
                'password'     => ['nullable','confirmed','min:8'],
                'phone'        => ['nullable','string','max:120'],
                'address1'     => ['nullable','string','max:120'],
                'address2'     => ['nullable','string','max:120'],
                'postcode'     => ['nullable','string','max:10'],
                'city'         => ['nullable','string','max:120'],
                'country_isocode' => ['nullable','string','max:2'],
            ]);

            // Upload photo (optional)
            if ($request->hasFile('photo')) {
                if (!empty($user->photo) && Storage::disk('public')->exists($user->photo)) {
                    Storage::disk('public')->delete($user->photo);
                }
                $data['photo'] = $request->file('photo')->store('photos', 'public');
            }

            // Password (optional)
            if (!empty($data['password'])) {
                $data['password'] = Hash::make($data['password']);
            } else {
                unset($data['password']);
            }

            // Update the existing columns on the users table
            $user->fill([
                'name'         => $data['firstname'] ?? $user->name,
                'firstname'    => $data['firstname'] ?? $user->firstname,
                'lastname'     => $data['lastname']  ?? $user->lastname,
                'email'        => $data['email']     ?? $user->email,
                'password'     => $data['password']  ?? $user->password,
                'phone'        => $data['phone']     ?? $user->phone,
                'address1'     => $data['address1']  ?? $user->address1,
                'address2'     => $data['address2']  ?? $user->address2,
                'postcode'     => $data['postcode']  ?? $user->postcode,
                'city'         => $data['city']      ?? $user->city,
                'country_isocode' => $data['country_isocode'] ?? $user->country_isocode,

            ]);

            if (isset($data['password'])) $user->password = $data['password'];
            if (isset($data['photo']))   $user->photo   = $data['photo'];

            $user->save();

            return back()->with('ok', 'Profil mis à jour.');
        }
    }
