<?php
    // app/Models/Itinerary.php
    namespace App\Models;

    use Illuminate\Database\Eloquent\Model;
    use Illuminate\Database\Eloquent\Casts\Attribute;
    use Illuminate\Support\Carbon;

    class Itinerary extends Model
    {

        protected $fillable = [
            'name',
            'owner_user_id',
            'assigned_user_id',
            'society_id',
            'scheduled_date',
        ];
    
        
        protected $casts = [
            // si la colonne est de type DATE -> 'date'
            // si TIMESTAMP/DATETIME -> 'datetime'
            'scheduled_date' => 'date',
        ];

 


        // Société concernée
        public function society()
        {
            return $this->belongsTo(\App\Models\Society::class, 'society_id');
        }
    
      
        // Utilisateurs associés à cet itinéraire (table pivot itinerary_user)
        public function users()
        {
            return $this->belongsToMany(\App\Models\User::class, 'itinerary_user', 'itinerary_id', 'user_id');
        }



        public function manager()
        {
            // assigned_user_id = admin(1) or pro admin (role_id=2)
            return $this->belongsTo(User::class, 'assigned_user_id');
        }

    
        public function owner()   
        { 
            return $this->belongsTo(User::class, 'owner_user_id'); 
        }

        public function assignee()
        { 
            return $this->belongsTo(User::class, 'assigned_user_id'); 
        }

        // Each row = one stop, duplicates allowed, ordered
        public function items()
        {
            return $this->hasMany(\App\Models\ItineraryPoint::class)->orderBy('sort_order');
        }
  


        public function itineraryPoints()
        {
            return $this->hasMany(ItineraryPoint::class, 'itinerary_id')->orderBy('order');
        }


        public function assignedUsers()
        {
            return $this->belongsToMany(User::class, 'itinerary_user', 'itinerary_id', 'user_id')
                ->withTimestamps();
        }


        // Admin ou Admin Pro en charge de cet itinéraire
        public function assignedUser()
        {
            return $this->belongsToMany(User::class, 'itinerary_user')
                ->withTimestamps();
        }

   
        public function itemsUser()
        {
            // Modèle supposé: App\Models\ItineraryPointUser
            return $this->hasMany(ItineraryPointUser::class, 'itinerary_id');
        }


        
        public function points()
        {
            return $this->belongsToMany(Point::class, 'itinerary_points', 'itinerary_id', 'point_id')
                ->withPivot(['sort_order', 'arrival_at', 'planned_at', 'done', 'validated_at', 'rescheduled_at', 'comment'])
                ->withTimestamps()
                ->orderBy('itinerary_points.sort_order');
        }



        protected function scheduledDate(): Attribute
        {
            return Attribute::get(fn ($value) => $value ? Carbon::parse($value) : null);
        }



        public function templateItems() // le gabarit sur itinerary_points
        {
            return $this->belongsToMany(Point::class, 'itinerary_points', 'itinerary_id', 'point_id')
                ->withPivot(['sort_order','planned_at','arrival_at','done','validated_at','rescheduled_at','comment'])
                ->withTimestamps()
                ->orderBy('itinerary_points.sort_order');
        }






     }





