<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use App\Models\User;


class Point extends Model
{
    protected $fillable = [
        'user_id',
        'name',
        'latitude',
        'longitude',
        'altitude',
        'floor',
        'favoris',
        'comment',
        'address1',
        'address2',
        'postcode',
        'city',
        'country_isocode',
        'photo',
        'serial',
        'qrcode',
    ];


    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }
    
    public function owner()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function scopeVisibleFor(Builder $query, ?User $user): Builder
    {
        // If the user is not authenticated, return nothing
        if (!$user) {
            return $query->whereRaw('1 = 0');
        }

        if ($user->id === 1) { // Super admin
            return $query;
        }

        $now = now();

        $query->addSelect('points.*');

        return $query->where(function ($q) use ($user, $now) {
            $q->where('user_id', $user->id)
              ->orWhere('user_id', 0) // 👈 include public points
              ->orWhereIn('id', function ($sub) use ($user, $now) {
                  $sub->from('sharingdb')->select('point_id')
                      ->where('target_user_id', $user->id)
                      ->where(function ($sub) use ($now) {
                          $sub->whereNull('start_at')->orWhere('start_at', '<=', $now);
                      })
                      ->where(function ($sub) use ($now) {
                          $sub->whereNull('end_at')->orWhere('end_at', '>=', $now);
                      });
              });
        });

     }



     
    public function sharings()
    {
        return $this->hasMany(\App\Models\Sharing::class, 'point_id');
    }




    protected static function booted()
    {
        static::creating(function ($point) {
            if (empty($point->serial)) {
                // Simple example: prefix + timestamp
                $point->serial = 'VER-' . strtoupper(uniqid());
            }
        });
    }




    public function favoredBy()
    {
        return $this->belongsToMany(User::class, 'users_points_favorite')
                    ->withTimestamps();
    }

}
