<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use App\Models\Role;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Contracts\Auth\MustVerifyEmail;       
use Illuminate\Support\Str;
use Illuminate\Auth\Events\Registered;


class User extends Authenticatable implements MustVerifyEmail
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable;

    use HasApiTokens;

    
    
    protected $fillable = [
        'role_id', 'firstname','lastname','name', 'email','phone','photo','address1','address2',
        'postcode','city','country_isocode','password'
    ];



    protected $casts = [
        // ...
        'blocked' => 'datetime',
    ];




    public function isBlocked(): bool
    {
        return !is_null($this->blocked);
    }





    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */

    public function points()
    {
        return $this->hasMany(\App\Models\Point::class);
    }




    public function setEmailAttribute($value)
    {
        $this->attributes['email'] = Str::lower(trim($value));
    }





    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];





    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }



    

    public function isAdmin(): bool
    {
        // adapte la condition à ton schéma (ex: role_id === 1)
        return (int)($this->role_id ?? 0) === 1;
    }



 

    public function role()
    {
        return $this->belongsTo(\App\Models\Role::class, 'role_id');
    }



    
    public function connections()
    {
        return $this->hasMany(\App\Models\UserConnection::class);
    }



    public function hasRoleId(int $rid): bool
    {
        return (int)($this->role_id ?? 0) === $rid
            || $this->roles()->where('id',$rid)->exists();
    }



    /**
     * Latest connection record.
     */
    public function lastConnection()
    {
        return $this->hasOne(\App\Models\UserConnection::class)->latestOfMany('connected_at');
    }
    



    /**
     * Consider the user “online” if their last connection has no disconnected_at timestamp.
     * Optionally add an activity TTL (e.g., 10 minutes) if you record heartbeats.
     */
    public function getIsOnlineAttribute(): bool
    {
        $last = $this->lastConnection; // via with('lastConnection')
        return $last && is_null($last->disconnected_at);
    }
    



    /**
     * Return the last-seen timestamp.
     * - If online: connected_at
     * - Otherwise: the most recent disconnected_at (fallback to connected_at)
     */
    public function getLastSeenAttribute(): ?\Carbon\Carbon
    {
        $last = $this->lastConnection;
        if (!$last) return null;
    
        return $this->is_online
            ? $last->connected_at
            : ($last->disconnected_at ?: $last->connected_at);
    }




    /*  Societies the user belongs to.
     */

    public function societies()
    {
        return $this->belongsToMany(
            \App\Models\Society::class,
            'societydb_user',      // pivot
            'user_id',             // FK locale
            'society_id'           // FK liée
        )->withTimestamps();
    }



    
    /** Return the first linked society id (Admin Pro must have exactly one) */
    public function primarySocietyId(): ?int
    {
        // Assuming exactly one link for role_id=2
        $rel   = $this->societies();                 // BelongsToMany relation
        $table = $rel->getRelated()->getTable();     // e.g. 'societydb'
        // Ask specifically for "<relatedTable>.id" to avoid ambiguous 'id'
        return $rel->select("$table.id")->value('id');
    }



    
    public function scopeAdminsPro($q)
    {
        return $q->where('role_id', 2);
    }
    
    public function scopePros($q)
    {
        return $q->where('role_id', 4);
    }


    public function itineraries()
    {
        return $this->belongsToMany(\App\Models\Itinerary::class, 'itinerary_user')
            ->withTimestamps();
    }




    public function favoritePoints() 
    {
        return $this->belongsToMany(\App\Models\Point::class, 'users_points_favorite', 'user_id', 'point_id')
                    ->withTimestamps();
    }



    public function itinerariesAssigned()
    {
        // pivot: itinerary_user (itinerary_id, user_id)
        return $this->belongsToMany(\App\Models\Itinerary::class, 'itinerary_user', 'user_id', 'itinerary_id')
            ->withTimestamps();
    }



    public function itineraryItems()
    {
        return $this->hasMany(\App\Models\ItineraryPointUser::class);
    }



}
