<?php

namespace App\Services;

use App\Models\Point;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Http\Controllers\PointController;

// BaconQrCode (v2)
use BaconQrCode\Common\ErrorCorrectionLevel;
use BaconQrCode\Writer;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;

class PointQr
{
    /* ============================================================
     * Helpers — SVG cleanup & utilities
     * ============================================================ */

    /** Remove BOM and any XML declaration. */
    
    private static function svgStripBomAndXmlDecl(string $svg): string
    {
        $svg = preg_replace('/^\xEF\xBB\xBF/', '', $svg) ?? $svg; // Remove BOM
        $svg = ltrim($svg);
        $svg = preg_replace('/<\?xml[^>]*\?>\s*/i', '', $svg) ?? $svg; // Remove XML declaration
        return $svg;
    }







    /** Enforce a single <svg> root and extract the inner content when given nested SVG (logo). */
    private static function svgEnsureRootOnlyOnce(string $svg): string
    {
        if (preg_match('/<svg[^>]*>(.*)<\/svg>/is', $svg, $m)) {
            return trim($m[1]);
        }
        return trim($svg);
    }







    /** Apply width/height + viewBox attributes to the <svg> tag. */
    private static function forceViewBoxAndSize(string $svg, int $w, int $h): string
    {
        $svg = self::svgStripBomAndXmlDecl($svg);
        // Inject width/height/viewBox on the first <svg> tag
        $svg = preg_replace_callback('/<svg\b([^>]*)>/i', function ($m) use ($w, $h) {
            $attrs = preg_replace('/\s(width|height|viewBox)\s*=\s*"[^"]*"/i', '', $m[1]);
            // shape-rendering improves readability for some scanners
            if (!preg_match('/\bshape-rendering=/i', $attrs)) {
                $attrs .= ' shape-rendering="crispEdges"';
            }
            return '<svg'.$attrs.' width="'.$w.'" height="'.$h.'" viewBox="0 0 '.$w.' '.$h.'">';
        }, $svg, 1);
        return $svg;
    }






    /** Convert a filesystem path (public/ or storage/public) to a data URI (png/jpg/svg). */
    private static function logoToDataUri(string $path): ?string
    {
        $abs = null;
        if (is_file(public_path(ltrim($path, '/')))) {
            $abs = public_path(ltrim($path, '/'));
        } elseif (Storage::disk('public')->exists($path)) {
            $abs = Storage::disk('public')->path($path);
        }
        if (!$abs) return null;
    
        $ext  = strtolower(pathinfo($abs, PATHINFO_EXTENSION));
        $mime = match ($ext) {
            'svg' => 'image/svg+xml',
            'png' => 'image/png',
            'jpg', 'jpeg' => 'image/jpeg',
            default => null,
        };
        if (!$mime) return null;
    
        $raw = @file_get_contents($abs);
        if ($raw === false) return null;
    
        return 'data:'.$mime.';base64,'.base64_encode($raw);
    }







    /** Inject the logo (data URI) in the center, with an optional rounded white background. */
    private static function embedCenteredLogo(string $svg, string $logoDataUri, float $ratio = 0.14): string
    {
        // Logical size inferred from the viewBox
        $S = 512;
        if (preg_match('/viewBox\s*=\s*"0\s+0\s+(\d+(?:\.\d+)?)\s+(\d+(?:\.\d+)?)"/i', $svg, $m)) {
            $S = (int) round(min((float)$m[1], (float)$m[2]));
        }
        $logo = (int) round($S * $ratio);
        $x = (int) round(($S - $logo)/2);
        $y = (int) round(($S - $logo)/2);
    
        // Slightly larger white background
        $bg  = (int) round($logo * 1.22);
        $bx  = (int) round(($S - $bg)/2);
        $by  = (int) round(($S - $bg)/2);
        $rx  = max(4, (int) round($bg * 0.08));
    
        $overlay = sprintf(
            '<g id="logo_overlay" pointer-events="none">
                <rect x="%1$d" y="%2$d" width="%3$d" height="%3$d" rx="%4$d" ry="%4$d" fill="#fff"/>
                <image href="%5$s" x="%6$d" y="%7$d" width="%8$d" height="%8$d" preserveAspectRatio="xMidYMid meet"/>
             </g>',
            $bx, $by, $bg, $rx,
            htmlspecialchars($logoDataUri, ENT_QUOTES, 'UTF-8'),
            $x, $y, $logo
        );
    
        return preg_replace('/<\/svg>\s*$/i', $overlay.'</svg>', $svg, 1);
    }






    /* ============================================================
     * QR generation (BaconQrCode)
     * ============================================================ */

    /** Generate a raw QR (SVG string) with ECC=H and margin 4. */
    private static function makeRawQrSvg(string $text, int $size = 512, int $margin = 4): string
    {
        $renderer = new ImageRenderer(
            new RendererStyle($size, $margin, null, null, null, ErrorCorrectionLevel::H()),
            new SvgImageBackEnd()
        );
        $writer = new Writer($renderer);
        return $writer->writeString($text);
    }





    /** Simple QR (clean SVG). */
    public static function makeSvg(string $url, int $size = 512): string
    {
        $svg = self::makeRawQrSvg($url, $size);
        $svg = self::svgStripBomAndXmlDecl($svg);
        $svg = self::forceViewBoxAndSize($svg, $size, $size);
        return $svg;
    }






    /** QR with a centered logo (clean SVG). */
    public static function makeSvgWithLogo(string $url, string $logoRelPath = '/assets/images/V_300.png', int $size = 512, float $logoRatio = 0.14): string
    {
        $qr = self::makeRawQrSvg($url, $size, 4);
        $qr = self::forceViewBoxAndSize($qr, $size, $size);
    
        if ($logo = self::logoToDataUri($logoRelPath)) {
            $qr = self::embedCenteredLogo($qr, $logo, $logoRatio);
        }
        // Final sanitation pass
        return self::svgStripBomAndXmlDecl($qr);
    }







    /* ============================================================
     * Integration with the Point model
     * ============================================================ */

    /** Ensure a serial exists in the database. */
    private static function ensureSerial(Point $p): void
    {
        $raw = $p->getAttributes()['serial'] ?? null;
        if (!empty($raw)) return;

        $p->serial = PointController::makeSerial($p);
        $p->save();
    }







    /**
     * Generate and store an **SVG** QR (with logo) for a Point.
     * - writes the file into storage/app/public/qrcodes
     * - updates $point->qrcode (e.g. "qrcodes/point_12-my-location.svg")
     * Returns the stored relative path.
     */
    public static function generateFor(Point $point, ?string $logoRel = 'assets/images/V_300.svg', int $size = 512, float $logoRatio = 0.18): ?string
    {
        try {
            self::ensureSerial($point);

            // Localized public URL (absolute)
            // $url = route('points.public.readonly', [
            //     'locale' => app()->getLocale() ?: 'fr',
            //     'token'  => $point->serial,
            // ], true);
            
            // Non-localized public URL (absolute)
            $url = route('points.wx', ['token' => $point->serial], true);

            // Final SVG content
            $svg = self::makeSvgWithLogo($url, $logoRel, $size, $logoRatio);

            // File name & public storage
            $filename = "V{$point->id}-{$point->serial}.svg";
            $file = "qrcodes/{$filename}";

            Storage::disk('public')->put($file, $svg);
            $point->update(['qrcode' => $file]);

            Log::debug('QR SVG generated', [
                'point_id' => $point->id,
                'path'     => $file,
                'url'      => $url,
            ]);

            return $file;
        } catch (\Throwable $e) {
            Log::error('QR generation failed', [
                'point_id' => $point->id ?? null,
                'err'      => $e->getMessage(),
            ]);
            return null;
        }
    }





    /**
     * Generate a “simple” SVG (without a logo) for a point.
     * Saves it under storage/app/public/qrcodes/... and updates $point->qrcode
     */
    public static function generateSvgSimple(Point $point): string
    {
        // 1) Ensure a serial is present
        if (empty($point->serial)) {
            $point->serial = \App\Http\Controllers\PointController::makeSerial($point);
            $point->save();
        }

        // 2) Short non-localized URL
        $url = route('points.wx', ['token' => $point->serial], true);

        // 3) SVG QR without a logo (logical size 512, margin 4)
        $renderer = new ImageRenderer(
            new RendererStyle(512, 4),
            new SvgImageBackEnd()
        );
        $writer = new Writer($renderer);
        $svg = $writer->writeString($url);

        // 4) File name & public storage
        // $slug  = Str::slug($point->name ?: 'qr');
        // $slug  = Str::limit($slug, 60, '');
        // $fname = "point_{$point->id}-{$slug}.svg";
        // $path  = "qrcodes/{$fname}";               // => storage/app/public/qrcodes/…
 
        $filename = "V{$point->id}-{$point->serial}.svg";
        $path = "qrcodes/{$filename}";


        Storage::disk('public')->put($path, $svg);
        $point->update(['qrcode' => $path]);

        Log::debug('QR SVG (simple) generated', [
            'point_id' => $point->id,
            'path'     => $path,
            'url'      => $url,
        ]);

        return $path;
    }
}
