{{-- resources/views/admin/itins/create.blade.php --}}
@extends('layouts.base')

@section('title', __('Create an itinerary'))

@section('content')

  {{-- Minimal styles for a clean UI --}}
    <style>
        .page-title { margin:0 0 12px; }
        .max-w { max-width: 1200px; margin-inline:auto; }
        .ul-reset { list-style:none; padding:0; }
        .grid-3 { display:grid; grid-template-columns: 1fr 1fr 1fr; gap:16px; }
        .twocol { display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
        .panel { border:var(--border); border-radius:12px; overflow:hidden;  }
        .panel-hd { padding:10px 12px; border-bottom:var(--border);  }
        .list { padding:10px; min-height:220px; border-top:var(--border); }
        .row { display:flex; align-items:center; justify-content:space-between; gap:8px; padding:8px 10px; border:1px solid var(--border-color,#1f2430); border-radius:8px; margin-bottom:8px;  }
        .row .btn.small { padding:2px 10px !important; }
        .mb-6 { margin-bottom:6px; }
    </style>

 
    <div class="wrap">
    <h2 class="page-title">{{ __('Create an itinerary') }}</h2>

    @if($errors->any())
        <div class="alert danger">
        <ul class="ul-reset">
            @foreach($errors->all() as $err)<li>{{ $err }}</li>@endforeach
        </ul>
        </div>
    @endif

    <form method="POST" action="{{ route('admin.itins.store', ['locale'=>app()->getLocale()]) }}">
        @csrf

        <div class="card max-w">
            <div class="card-body">

                {{-- Step 1: Name --}}
                <div class="form-field">
                <label for="name">{{ __('Name') }} *</label>
                <input id="name" type="text" name="name" required value="{{ old('name') }}" placeholder="Ex: Tournée Sud-Est">
                <small class="muted">{{ __('A unique itinerary name.') }}</small>
                </div>

                <div class="grid-2">
                    {{-- Step 2: Company --}}
                    <div class="form-field">
                        <label for="society_id">{{ __('Company') }} *</label>
                        {{-- Select société (filtré côté contrôleur) --}}
                        <select id="society_id" name="society_id" required>
                            <option value="" disabled selected>{{ __('Select…') }}</option>
                            @foreach($societies as $s)
                                <option value="{{ $s->id }}">{{ $s->name }}</option>
                            @endforeach
                        </select>

                    </div>

                    {{-- Step 3: Pro Admin (depends on company) --}}
                    <div class="form-field">
                        <label for="assigned_user_id">{{ __('Pro admin') }} *</label>
                        {{-- Select admin pro (dyn.) --}}
                        <select id="assigned_user_id" name="assigned_user_id" required disabled>
                            <option value="">{{ __('Select a company first') }}</option>
                        </select>
                    </div>

                 </div>

                {{-- Step 4: Waypoints for the chosen society + shared to chosen admin --}}
                <div class="form-field" style="margin-top:10px;">
                    <label>{{ __('Waypoints') }}</label>
                    <div class="muted mb-6">
                        {{ __('Pick a company, then a Pro admin to load waypoints of the company and those shared to that admin. Add and drag to reorder them.') }}
                    </div>

                    <div class="twocol">
                        {{-- Available points --}}
                        <div class="panel">
                            <div class="panel-hd">
                                <h4>{{ __('Available') }}</h4>
                            </div>
                            <div id="available-points" class="list">
                                <div class="muted" data-empty>{{ __('Select a company and a Pro admin.') }}</div>
                            </div>
                        </div>

                        {{-- Selected (ordered) --}}
                        <div class="panel">
                            <div class="panel-hd">
                                <h4>{{ __('Selected (drag to reorder)') }}</h4>
                            </div>
                            <div id="selected-points" class="list droplist">
                                <div class="muted" data-empty>{{ __('No waypoint selected yet.') }}</div>
                            </div>
                            {{-- Hidden container to mirror DOM order --}}
                            <div id="point-ids-hidden"></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="card-footer actions">
                <a class="btn ghost" href="{{ route('admin.itins.index', ['locale'=>app()->getLocale()]) }}">{{ __('Back') }}</a>
                <button class="btn primary" type="submit">{{ __('Create') }}</button>
            </div>
        </div>
    </form>

 
    @php
        $urlAdmins  = url('/admin/itineraries/admins-for-society');
        $urlPoints  = url('/admin/itineraries/points-for-society');
    @endphp

    @push('scripts')
        <script>
            document.addEventListener('DOMContentLoaded', () => {
                const selectSoc   = document.getElementById('society_id');
                const selectAdmin = document.getElementById('assigned_user_id');

                // URLs côté admin (non localisées)
                const urlAdmins = @json($urlAdmins);
                const urlPoints = @json($urlPoints);

                // Textes (on JSON-encode côté Blade une fois, puis on réutilise)
                const T = {
                    select: @json(__('Select…')),
                    selectCompanyFirst: @json(__('Select a company first')),
                    loading: @json(__('Loading…')),
                    noProForCompany: @json(__('No pro admin for this company')),
                    errAdmins: @json(__('Error loading admins')),
                    noWpForCombo: @json(__('No waypoint for this company / admin.')),
                    errWps: @json(__('Unable to load waypoints')),
                    noneSelected: @json(__('No waypoint selected yet.')),
                    selectCompanyAndAdmin: @json(__('Select a company and a Pro admin.')),
                };

                // Helpers
                function fillAdminSelect(items) {
                    if (!Array.isArray(items) || items.length === 0) {
                        selectAdmin.innerHTML = `<option value="">${T.noProForCompany}</option>`;
                        selectAdmin.disabled  = true;
                        return;
                    }
                    selectAdmin.disabled = false;
                    selectAdmin.innerHTML = `<option value="">${T.select}</option>`;
                    for (const u of items) {
                        const opt  = document.createElement('option');
                        const name = `${u.firstname ?? ''} ${u.lastname ?? ''}`.trim() || `#${u.id}`;
                        opt.value  = u.id;
                        opt.textContent = name;
                        selectAdmin.appendChild(opt);
                    }
                    if (items.length === 1) {
                        const only = items[0];
                        const onlyId = Number(only?.id);
                        if (Number.isInteger(onlyId) && onlyId > 0) {
                            const desired = String(onlyId);
                            if (selectAdmin.value !== desired) {
                                selectAdmin.value = desired;
                                selectAdmin.dispatchEvent(new Event('change', { bubbles: true }));
                            }
                        }
                    }
                }

                function resetWaypointsUI({ clearAvailable = false } = {}) {
                    const avail = document.getElementById('available-points');
                    const sel   = document.getElementById('selected-points');
                    const hidden= document.getElementById('point-ids-hidden');
                    if (clearAvailable && avail) {
                        avail.innerHTML  = `<div class="muted" data-empty>${T.selectCompanyAndAdmin}</div>`;
                    }
                    if (sel) {
                        sel.innerHTML    = `<div class="muted" data-empty>${T.noneSelected}</div>`;
                    }
                    if (hidden) hidden.innerHTML = '';
                }

                async function fetchJson(url) {
                    // Petit wrapper sûr : log la réponse texte si pas JSON, évite les "Unexpected token <"
                    const res  = await fetch(url, {
                        headers: {
                            'Accept': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest'
                        },
                        credentials: 'same-origin'
                    });

                    const ctype = res.headers.get('content-type') || '';
                    const txt   = await res.text();

                    if (!res.ok) {
                        console.error(`[FETCH] ${url} → HTTP ${res.status}`, txt);
                        throw new Error(`HTTP ${res.status}`);
                    }

                    if (!ctype.includes('application/json')) {
                        // Probable HTML (login, 419, 500…)
                        console.error(`[FETCH] ${url} → Not JSON, got:`, ctype, txt);
                        throw new Error('Unexpected non-JSON response');
                    }

                    try {
                        return JSON.parse(txt);
                    } catch (e) {
                        console.error(`[FETCH] ${url} → JSON parse error`, e, txt);
                        throw e;
                    }
                }

                // 1) Changement de société → charge les admins pro
                selectSoc?.addEventListener('change', async () => {
                    const sid = selectSoc.value;

                    // UI état loading
                    selectAdmin.disabled  = true;
                    selectAdmin.innerHTML = `<option value="">${T.loading}</option>`;
                    resetWaypointsUI({ clearAvailable: true });

                    if (!sid) return;

                    try {
                        const data = await fetchJson(`${urlAdmins}?society_id=${encodeURIComponent(sid)}`);
                        console.log('adminsForSociety →', data);
                        fillAdminSelect(data);
                    } catch (e) {
                        console.error('adminsForSociety error:', e);
                        selectAdmin.innerHTML = `<option value="">${T.errAdmins}</option>`;
                        selectAdmin.disabled  = true;
                    }
                });

                // 2) Changement d’admin pro → charge les waypoints (société + partagés à cet admin)
                selectAdmin?.addEventListener('change', async () => {
                    const sid = selectSoc.value;
                    const aid = selectAdmin.value;
                    const avail = document.getElementById('available-points');
                    resetWaypointsUI();

                    if (avail) avail.innerHTML = `<div class="muted">${T.loading}</div>`;
                    if (!sid || !aid) {
                        if (avail) avail.innerHTML = `<div class="muted" data-empty>${T.selectCompanyAndAdmin}</div>`;
                        return;
                    }

                    try {
                        const list = await fetchJson(`${urlPoints}?society_id=${encodeURIComponent(sid)}&admin_id=${encodeURIComponent(aid)}`);

                        // Affichage
                        if (!Array.isArray(list) || list.length === 0) {
                            if (avail) avail.innerHTML = `<div class="muted">${T.noWpForCombo}</div>`;
                            return;
                        }

                        if (avail) {
                            avail.innerHTML = '';
                            list.forEach(p => {
                                const id = Number(p?.id);
                                if (!Number.isInteger(id) || id <= 0) return;

                                const row = document.createElement('div');
                                row.className = 'row';
                                row.draggable = true;
                                row.dataset.id = String(id);
                                row.innerHTML = `
                                <span>${p.name || ('#'+id)}</span>
                                <div style="display:flex; gap:6px;">
                                    <button type="button" class="btn small ghost add">{{ __('Add') }}</button>
                                    <button type="button" class="btn small danger remove" style="display:none;">{{ __('Remove') }}</button>
                                </div>`;
                                avail.appendChild(row);
                            });
                        }
                    } catch (err) {
                        console.error('pointsForSociety exception', err);
                        if (avail) avail.innerHTML = `<div class="alert danger">${T.errWps}</div>`;
                    }
                    });

                // (tu peux garder ici ton code de drag & drop + mise à jour des inputs hidden)
            });

            document.addEventListener('DOMContentLoaded', () => {
                const boxAvail  = document.getElementById('available-points');
                const boxSel    = document.getElementById('selected-points');
                const hidden    = document.getElementById('point-ids-hidden');
                const urlPoints = @json($urlPoints);

                if (!boxAvail || !boxSel || !hidden) return;

                // Helpers
                const t = {
                    noneSelected: '{{ __('No waypoint selected yet.') }}'
                };

                function ensureEmptyPlaceholder(container) {
                    if (!container.querySelector('[data-empty]') && container.querySelectorAll('.row').length === 0) {
                    container.innerHTML = `<div class="muted" data-empty>${t.noneSelected}</div>`;
                    }
                }
                function removeEmptyPlaceholder(container) {
                    const emp = container.querySelector('[data-empty]');
                    if (emp) emp.remove();
                }
                function isInSelected(id) {
                    return !!boxSel.querySelector(`.row[data-id="${CSS.escape(String(id))}"]`);
                }
                function updateHidden() {
                    hidden.innerHTML = '';
                    boxSel.querySelectorAll('.row[data-id]').forEach((row) => {
                    const raw = (row.dataset.id || '').trim();
                    const id  = Number(raw);
                    if (!Number.isInteger(id) || id <= 0) return;   // ← ignore invalides
                        const inp = document.createElement('input');
                        inp.type  = 'hidden';
                        inp.name  = 'point_ids[]';
                        inp.value = String(id);
                        hidden.appendChild(inp);
                    });
                }

                // Drag & drop (colonne de droite)
                let dragging = null;
                boxSel.addEventListener('dragstart', (e) => {
                    const row = e.target.closest('.row');
                    if (!row) return;
                    dragging = row;
                    e.dataTransfer.effectAllowed = 'move';
                });
                boxSel.addEventListener('dragover', (e) => {
                    if (!dragging) return;
                    e.preventDefault();
                    const rows = [...boxSel.querySelectorAll('.row')].filter(r => r !== dragging);
                    const after = rows.find(r => e.clientY < r.getBoundingClientRect().top + r.offsetHeight / 2);
                    if (after) boxSel.insertBefore(dragging, after);
                    else boxSel.appendChild(dragging);
                });
                boxSel.addEventListener('drop', () => { dragging = null; updateHidden(); });
                boxSel.addEventListener('dragend', () => { dragging = null; updateHidden(); });

                // Délégation des clics Add / Remove
                document.addEventListener('click', (e) => {
                    // ADD
                    const addBtn = e.target.closest('.add');
                    if (addBtn) {
                        const row = addBtn.closest('.row'); if (!row) return;
                        const id  = Number(row.dataset.id);
                        if (!Number.isInteger(id) || id <= 0) return; // ignore invalide
                        if (isInSelected(id)) return;

 
                        // toggle boutons
                        addBtn.style.display = 'none';
                        const rem = row.querySelector('.remove');
                        if (rem) rem.style.display = '';

                        removeEmptyPlaceholder(boxSel);
                        boxSel.appendChild(row);
                        updateHidden();
                        return;
                    }

                    // REMOVE
                    const remBtn = e.target.closest('.remove');
                    if (remBtn) {
                    const row = remBtn.closest('.row');
                    if (!row) return;

                    // toggle boutons
                    remBtn.style.display = 'none';
                    const add = row.querySelector('.add');
                    if (add) add.style.display = '';

                    boxAvail.appendChild(row);
                    ensureEmptyPlaceholder(boxSel);
                    updateHidden();
                    return;
                    }
                });

                // Sécurité : si on recharge des points “Available”, on (re)branche le draggable côté droite
                const observer = new MutationObserver(() => {
                    // quand la colonne droite change → recalcule les inputs cachés
                    updateHidden();
                });
                observer.observe(boxSel, { childList: true, subtree: true });
            });
        </script>
    @endpush
@endsection
