{{-- resources/views/admin/itins/edit.blade.php --}}
@extends('layouts.base')

@section('title', __('Edit itinerary'))

@section('content')
<style>
  .page-title { margin:0 0 12px; }
  .max-w { max-width: 1200px; margin-inline:auto; }
  .ul-reset { list-style:none; padding:0; }
  .twocol { display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
  .panel { border:var(--border); border-radius:12px; overflow:hidden; }
  .panel-hd { padding:10px 12px; border-bottom:var(--border);  }
  .list { padding:10px; min-height:220px; border-top:var(--border); }
  .row { display:flex; align-items:center; justify-content:space-between; gap:8px; padding:8px 10px; border:1px solid var(--border-color,#1f2430); border-radius:8px; margin-bottom:8px; }
  .row .btn.small { padding:2px 10px !important; }
  .mb-6 { margin-bottom:6px; }
</style>

@php
    $urlAdmins  = url('/admin/itineraries/admins-for-society');
    $urlPoints  = url('/admin/itineraries/points-for-society');
@endphp

<div class="wrap">
    <h2 class="page-title">{{ __('Edit itinerary') }}</h2>

    @if(session('ok'))
        <div class="ok">{{ session('ok') }}</div>
    @endif

    @if($errors->any())
        <div class="alert danger">
            <ul class="ul-reset">
                @foreach($errors->all() as $err)<li>{{ $err }}</li>@endforeach
            </ul>
        </div>
    @endif

    <form method="POST" action="{{ route('admin.itins.update', ['itinerary'=>$itinerary->id]) }}">
        @csrf
        @method('PUT')

        <div class="card max-w">
            <div class="card-body">

                {{-- Name --}}
                <div class="form-field">
                    <label for="name">{{ __('Name') }} *</label>
                    <input id="name" type="text" name="name" required value="{{ old('name', $itinerary->name) }}">
                    <small class="muted">{{ __('A unique itinerary name.') }}</small>
                </div>

                <div class="twocol">
                    {{-- Company --}}
                    <div class="form-field">
                        <label for="society_id">{{ __('Company') }} *</label>
                        <select id="society_id" name="society_id" required>
                            <option value="" disabled>{{ __('Select…') }}</option>
                            @foreach($societies as $s)
                                <option value="{{ $s->id }}" @selected(old('society_id', $itinerary->society_id)==$s->id)>{{ $s->name }}</option>
                            @endforeach
                        </select>
                    </div>

                    {{-- Pro admin (dynamic) --}}
                    <div class="form-field">
                        <label for="assigned_user_id">{{ __('Pro admin') }}</label>
                        <select id="assigned_user_id" name="assigned_user_id" @disabled(!$preAssignedAdminId && !old('assigned_user_id'))>
                            <option value="">{{ __('Select a company first') }}</option>
                        </select>
                        <small class="muted">{{ __('Optional if you use multi-assignment via the checklist below.') }}</small>
                    </div>
                </div>

                {{-- Assigned users checklist (members of company, id > 1) --}}
                <div class="form-field">
                    <label>{{ __('Assigned users (company members)') }}</label>
                    <div id="company-users-box" style="display:grid; grid-template-columns: repeat(auto-fit,minmax(240px,1fr)); gap:8px;">
                        {{-- Rempli côté JS quand société change, et auto-chargé au chargement --}}
                    </div>
                </div>

                {{-- Waypoints --}}
                <div class="form-field" style="margin-top:10px;">
                    <label>{{ __('Waypoints') }}</label>
                    <div class="muted mb-6">
                        {{ __('Pick a company, then a Pro admin to load waypoints of the company and those shared to that admin. Add and drag to reorder them.') }}
                    </div>

                    <div class="twocol">
                        {{-- Available --}}
                        <div class="panel">
                            <div class="panel-hd"><h4>{{ __('Available') }}</h4></div>
                            <div id="available-points" class="list">
                                <div class="muted" data-empty>{{ __('Select a company and a Pro admin.') }}</div>
                            </div>
                        </div>

                        {{-- Selected --}}
                        <div class="panel">
                            <div class="panel-hd"><h4>{{ __('Selected (drag to reorder)') }}</h4></div>
                            <div id="selected-points" class="list droplist">
                                {{-- Pré-rempli avec $selectedPoints --}}
                                @forelse($selectedPoints as $p)
                                    <div class="row" draggable="true" data-id="{{ $p['id'] }}">
                                        <span>{{ $p['name'] ?? ('#'.$p['id']) }}</span>
                                        <div style="display:flex; gap:6px;">
                                            <button type="button" class="btn small ghost add" style="display:none;">{{ __('Add') }}</button>
                                            <button type="button" class="btn small danger remove">{{ __('Remove') }}</button>
                                        </div>
                                    </div>
                                    @empty
                                    <div class="muted" data-empty>{{ __('No waypoint selected yet.') }}</div>
                                @endforelse
                            </div>
                            <div id="point-ids-hidden">
                                @foreach($selectedPoints as $p)
                                    <input type="hidden" name="point_ids[]" value="{{ $p['id'] }}">
                                @endforeach
                            </div>
                        </div>
                    </div>
                </div>

            </div>

            <div class="card-footer actions">
                <a class="btn ghost" href="{{ route('admin.itins.index') }}">{{ __('Back') }}</a>
                <button class="btn primary" type="submit">{{ __('Save') }}</button>
            </div>
        </div>
    </form>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', () => {
  const selectSoc     = document.getElementById('society_id');
  const selectAdmin   = document.getElementById('assigned_user_id');
  const boxUsers      = document.getElementById('company-users-box');
  const boxAvail      = document.getElementById('available-points');
  const boxSel        = document.getElementById('selected-points');
  const hidden        = document.getElementById('point-ids-hidden');
  const urlAdmins = @json($urlAdmins);
  const urlPoints = @json($urlPoints);

  const T = {
    select: @json(__('Select…')),
    selectCompanyFirst: @json(__('Select a company first')),
    loading: @json(__('Loading…')),
    noProForCompany: @json(__('No pro admin for this company')),
    errAdmins: @json(__('Error loading admins')),
    noWpForCombo: @json(__('No waypoint for this company / admin.')),
    errWps: @json(__('Unable to load waypoints')),
    noUsersForCompany: @json(__('No users found for this company (excluding root).')),
    errUsers: @json(__('Unable to load users for this company.')),
    noneSelected: @json(__('No waypoint selected yet.')),
    selectCompanyAndAdmin: @json(__('Select a company and a Pro admin.')),
  };

  // Helpers
  function updateHidden() {
    hidden.innerHTML = '';
    boxSel.querySelectorAll('.row[data-id]').forEach((row) => {
    const raw = (row.dataset.id || '').trim();
    const id  = Number(raw);
    if (!Number.isInteger(id) || id <= 0) return;   // ← ignore invalides
        const inp = document.createElement('input');
        inp.type  = 'hidden';
        inp.name  = 'point_ids[]';
        inp.value = String(id);
        hidden.appendChild(inp);
    });
  }
  function ensureEmpty(container) {
    if (!container.querySelector('[data-empty]') && !container.querySelector('.row')) {
      container.innerHTML = `<div class="muted" data-empty>${T.noneSelected}</div>`;
    }
  }
  function removeEmpty(container) {
    const emp = container.querySelector('[data-empty]');
    if (emp) emp.remove();
  }
  function isInSelected(id) {
    return !!boxSel.querySelector(`.row[data-id="${CSS.escape(String(id))}"]`);
  }
  function resetWaypointsUI({ clearAvailable = false, clearSelected = true } = {}) {
    if (clearAvailable && boxAvail) {
      boxAvail.innerHTML = `<div class="muted" data-empty>${T.selectCompanyAndAdmin}</div>`;
    }
    if (clearSelected) {
      if (boxSel) {
        boxSel.innerHTML = `<div class="muted" data-empty>${T.noneSelected}</div>`;
      }
      if (hidden) hidden.innerHTML = '';
    }
  }

  // Drag & drop (selected column)
  let dragging = null;
  boxSel.addEventListener('dragstart', (e) => {
    const row = e.target.closest('.row'); if (!row) return;
    dragging = row; e.dataTransfer.effectAllowed = 'move';
  });
  boxSel.addEventListener('dragover', (e) => {
    if (!dragging) return;
    e.preventDefault();
    const rows = [...boxSel.querySelectorAll('.row')].filter(r => r !== dragging);
    const after = rows.find(r => e.clientY < r.getBoundingClientRect().top + r.offsetHeight/2);
    if (after) boxSel.insertBefore(dragging, after); else boxSel.appendChild(dragging);
  });
  boxSel.addEventListener('drop',  () => { dragging = null; updateHidden(); });
  boxSel.addEventListener('dragend', () => { dragging = null; updateHidden(); });

  // Add / Remove
  document.addEventListener('click', (e) => {
    const addBtn = e.target.closest('.add');
    if (addBtn) {
        const row = addBtn.closest('.row'); if (!row) return;
        const id  = Number(row.dataset.id);
        if (!Number.isInteger(id) || id <= 0) return; // ignore invalide
        if (isInSelected(id)) return;

        addBtn.style.display = 'none';
        const rem = row.querySelector('.remove'); if (rem) rem.style.display = '';
        removeEmpty(boxSel);
        boxSel.appendChild(row);
        updateHidden();
        return;
    }

    const remBtn = e.target.closest('.remove');
    if (remBtn) {
      const row = remBtn.closest('.row'); if (!row) return;
      remBtn.style.display = 'none';
      const add = row.querySelector('.add'); if (add) add.style.display = '';
      boxAvail.appendChild(row);
      ensureEmpty(boxSel);
      updateHidden();
      return;
    }
  });

  // Load admins for selected society
  async function fetchJson(url) {
    const res = await fetch(url, {
      headers: {'Accept':'application/json','X-Requested-With':'XMLHttpRequest'},
      credentials: 'same-origin'
    });
    const ctype = res.headers.get('content-type') || '';
    const txt   = await res.text();
    if (!res.ok) { console.error('HTTP',res.status,txt); throw new Error('HTTP '+res.status); }
    if (!ctype.includes('application/json')) { console.error('Not JSON',txt); throw new Error('Not JSON'); }
    return JSON.parse(txt);
  }

  async function loadAdmins(societyId, preselectId = null) {
    if (!societyId) return;
    selectAdmin.disabled = true;
    selectAdmin.innerHTML = `<option value="">${T.loading}</option>`;
    try {
      const list = await fetchJson(`${urlAdmins}?society_id=${encodeURIComponent(societyId)}`);
      if (!Array.isArray(list) || list.length === 0) {
        selectAdmin.innerHTML = `<option value="">${T.noProForCompany}</option>`;
        selectAdmin.disabled  = true;
        return;
      }
      selectAdmin.disabled = false;
      selectAdmin.innerHTML = `<option value="">${T.select}</option>`;
      const wantedPreselect = Number(preselectId);
      const hasPreselect = Number.isInteger(wantedPreselect) && wantedPreselect > 0;
      let resolvedId = null;
      list.forEach(u => {
        const opt = document.createElement('option');
        const name = `${u.firstname ?? ''} ${u.lastname ?? ''}`.trim() || `#${u.id}`;
        opt.value = u.id; opt.textContent = name;
        if (hasPreselect && Number(u.id) === wantedPreselect) {
          opt.selected = true;
          resolvedId = wantedPreselect;
        }
        selectAdmin.appendChild(opt);
      });
      if (resolvedId) {
        selectAdmin.value = String(resolvedId);
      } else if (list.length === 1) {
        const onlyId = Number(list[0]?.id);
        if (Number.isInteger(onlyId) && onlyId > 0) {
          const desired = String(onlyId);
          if (selectAdmin.value !== desired) {
            selectAdmin.dataset.autoload = '1';
            selectAdmin.value = desired;
            selectAdmin.dispatchEvent(new Event('change', { bubbles: true }));
          }
        }
      } else {
        selectAdmin.value = '';
      }
    } catch (e) {
      console.error('adminsForSociety error', e);
      selectAdmin.innerHTML = `<option value="">${T.errAdmins}</option>`;
      selectAdmin.disabled  = true;
    }
  }

  // Load users of company (checklist)
async function loadCompanyUsers(societyId) {
    if (!societyId) return;
    boxUsers.innerHTML = `<div class="muted">${T.loading}</div>`;
    try {
        const url = @json(route('admin.itins.usersForSociety')) + '?society_id=' + encodeURIComponent(societyId);
        const data = await fetchJson(url);
        if (!Array.isArray(data) || data.length === 0) {
        boxUsers.innerHTML = `<div class="muted">${T.noUsersForCompany}</div>`;
        return;
        }
        const assignedSet = new Set(@json($itinerary->assignedUsers->pluck('id')));
        boxUsers.innerHTML = '';
        data.forEach(u => {
            const id   = u.id;
            const name = `${u.firstname ?? ''} ${u.lastname ?? ''}`.trim();
            const email= u.email ?? '';
            const lbl  = document.createElement('label');
            lbl.className = 'chk';
            lbl.innerHTML = `
            <input type="checkbox" name="assigned_user_ids[]" value="${id}" ${assignedSet.has(id) ? 'checked' : ''}>
            <span>${name || '#'+id}</span>
            <small class="muted">— ${email}</small>
            `;
            boxUsers.appendChild(lbl);
        });
    } catch (e) {
        console.error('usersForSociety error', e);
        boxUsers.innerHTML = `<div class="alert danger">${T.errUsers}</div>`;
    }
}

  // Load points for society + admin
  async function loadPoints(societyId, adminId, { preserveSelection = false } = {}) {
    if (!boxAvail) return;

    if (!societyId || !adminId) {
      boxAvail.innerHTML = `<div class="muted" data-empty>${T.selectCompanyAndAdmin}</div>`;
      if (!preserveSelection) resetWaypointsUI({ clearAvailable: false, clearSelected: true });
      return;
    }

    boxAvail.innerHTML = `<div class="muted">${T.loading}</div>`;

    try {
      const list = await fetchJson(`${urlPoints}?society_id=${encodeURIComponent(societyId)}&admin_id=${encodeURIComponent(adminId)}`);
      if (!Array.isArray(list) || list.length === 0) {
        boxAvail.innerHTML = `<div class="muted">${T.noWpForCombo}</div>`;
        if (!preserveSelection) resetWaypointsUI({ clearAvailable: false, clearSelected: true });
        return;
      }

      boxAvail.innerHTML = '';
      list.forEach((p) => {
        const id = Number(p?.id);
        if (!Number.isInteger(id) || id <= 0) return;
        if (isInSelected(id)) return; // déjà sélectionné, ne pas dupliquer

        const row = document.createElement('div');
        row.className = 'row';
        row.draggable = true;
        row.dataset.id = String(id);
        row.innerHTML = `
            <span>${p.name || ('#'+id)}</span>
            <div style="display:flex; gap:6px;">
              <button type="button" class="btn small ghost add">{{ __('Add') }}</button>
              <button type="button" class="btn small danger remove" style="display:none;">{{ __('Remove') }}</button>
            </div>`;
        boxAvail.appendChild(row);
      });

      if (!boxAvail.querySelector('.row')) {
        boxAvail.innerHTML = `<div class="muted">${T.noWpForCombo}</div>`;
      }
    } catch (e) {
      console.error('pointsForSociety error', e);
      boxAvail.innerHTML = `<div class="alert danger">${T.errWps}</div>`;
      if (!preserveSelection) resetWaypointsUI({ clearAvailable: false, clearSelected: true });
    }
  }

  // Events
  selectSoc?.addEventListener('change', async () => {
    const sid = selectSoc.value;
    resetWaypointsUI({ clearAvailable: true, clearSelected: true });
    await Promise.all([
      loadAdmins(sid, null),
      loadCompanyUsers(sid),
    ]);
    if (boxAvail) boxAvail.innerHTML = `<div class="muted" data-empty>${T.selectCompanyAndAdmin}</div>`;
  });

  selectAdmin?.addEventListener('change', async () => {
    const sid = selectSoc.value;
    const aid = selectAdmin.value;
    const isAutoLoad = selectAdmin.dataset.autoload === '1';
    if (isAutoLoad) {
      delete selectAdmin.dataset.autoload;
    } else {
      resetWaypointsUI({ clearAvailable: false, clearSelected: true });
    }
    await loadPoints(sid, aid, { preserveSelection: isAutoLoad });
  });

  // Auto-préchargement au premier affichage (pré-sélection)
  (async () => {
    const sid = @json((int)old('society_id', (int)$itinerary->society_id));
    if (sid) {
      selectSoc.value = String(sid);
      await Promise.all([
        loadAdmins(sid, @json((int)old('assigned_user_id', (int)($preAssignedAdminId ?? 0)) ) || null),
        loadCompanyUsers(sid),
      ]);
      const preAdmin = selectAdmin.value || @json((string)($preAssignedAdminId ?? ''));
      if (preAdmin) {
        await loadPoints(sid, preAdmin, { preserveSelection: true });
      }
    }
  })();

  // Sur toute mutation de la colonne droite → MAJ des inputs
  const obs = new MutationObserver(updateHidden);
  obs.observe(boxSel, { childList: true, subtree: true });
});
</script>
@endpush
@endsection
