@extends('layouts.base')
@section('title', __('Itinerary map'))

@section('content')
    <div class="wrap">
        @if(session('ok')) <div class="ok">{{ session('ok') }}</div> @endif

        <h2 style="margin-bottom:.5rem">{{ $itinerary->name }}</h2>
        <div class="muted" style="margin-bottom:1rem;">
            {{ __('Assigned to') }} :
            {{ $itinerary->assignee->lastname ?? '' }} {{ $itinerary->assignee->firstname ?? '' }}
            @if($itinerary->scheduled_date) · {{ __('Date') }} : {{ $itinerary->scheduled_date }} @endif
            @if($itinerary->validated_date) · ✅ {{ __('Validated at') }} : {{ $itinerary->validated_date }} @endif
        </div>

        {{-- Carte --}}
        <div id="map" style="width:100%; height:420px; border:1px solid #1f2430; border-radius:10px; margin-bottom:14px;"></div>
  
        {{-- Buttons to export route --}}
        <div class="actions" style="display:flex; gap:.5rem; margin:10px 0;">
        <button id="btn-apple" class="btn small ghost"> Apple Maps</button>
        <button id="btn-google" class="btn small ghost">Google Maps</button>
        </div>

        {{-- Liste + progression --}}
        @php
            $locale = app()->getLocale();
            $isAdmin = (int)(auth()->id() ?? 0) === 1;

            $saveProgressUrl = $isAdmin
                ? route('admin.itins.progress.save', ['itinerary'=>$itinerary->id])
                : route('pro.itins.progress.save', ['locale'=>$locale, 'itinerary'=>$itinerary->id]);
        @endphp
        <form method="POST" action="{{ $saveProgressUrl }}">
            @csrf

            <ul style="list-style:none; padding:0; margin:0 0 12px 0;">
            @forelse($itinerary->items as $item)
                @php
                    $pt = $item->point;
                    $label = trim(($pt->name ?? '').($pt->city ? ' — '.$pt->city : ''));
                @endphp
            <div class="row">
            <label>
                <input type="checkbox" name="items[{{ $item->id }}][done]" {{ (int)$item->done === 1 ? 'checked' : '' }}>
                {{ $item->point?->name ?? '—' }}
            </label>

            <input type="datetime-local " name="items[{{ $item->id }}][arrival_at]"
                    value="{{ optional($item->arrival_at)->format('Y-m-d\TH:i') }}">
            <input type="datetime-local" name="items[{{ $item->id }}][validated_at]"
                    value="{{ optional($item->validated_at)->format('Y-m-d\TH:i') }}">
            <input type="datetime-local" name="items[{{ $item->id }}][rescheduled_at]"
                    value="{{ optional($item->rescheduled_at)->format('Y-m-d\TH:i') }}">
            <input type="text" name="items[{{ $item->id }}][comment]" value="{{ $item->comment }}">
            </div>
           @empty
                <li class="alert">{{ __('No waypoint in this itinerary') }}</li>
            @endforelse
            </ul>

            <div class="actions" style="display:flex; gap:.5rem;">
                <button class="btn primary" type="submit">{{ __('Save progress') }}</button>
            </div>

        </form>
  
        {{-- Bouton valider l’itinéraire --}}
        <form method="POST"
                action="{{ auth()->id()===1
                            ? route('admin.itins.validate', $itinerary)
                            : route('pro.itins.validate', ['locale'=>app()->getLocale(), 'itinerary'=>$itinerary->id]) }}"
                onsubmit="return confirm('{{ __('Validate this itinerary?') }}')">
            @csrf
            <button class="btn success" type="submit">
            {{ __('Validate itinerary') }}
            </button>
        </form>

        <a class="btn"
            href="{{ auth()->id()===1
                    ? route('admin.itins.index')
                    : route('pro.itins.index', ['locale'=>app()->getLocale()]) }}">
            {{ __('Back') }}
        </a>
    </div>

    {{-- Leaflet + Routing Machine --}}
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css"/>
    <link rel="stylesheet" href="https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.css"/>
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js" defer></script>
    <script src="https://unpkg.com/leaflet-routing-machine@3.2.12/dist/leaflet-routing-machine.min.js" defer></script>
    <script>
        document.addEventListener('DOMContentLoaded', () => {
            const items = {!! $itinerary->items
                ->map(function($it){
                    return [
                        'name'  => $it->point->name,
                        'lat'   => (float) $it->point->latitude,
                        'lng'   => (float) $it->point->longitude,
                        'city'  => $it->point->city,
                        'order' => (int) $it->sort_order,
                    ];
                })
                ->values()
                ->toJson(JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES) !!};

            if (!items.length) return;

            // --- Leaflet base map ---
            const map = L.map('map').setView([items[0].lat, items[0].lng], 12);
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                maxZoom: 19, attribution: '&copy; OpenStreetMap'
            }).addTo(map);

            // Place markers with popup order
            const wpts = items.map(p => L.latLng(p.lat, p.lng));
            items.forEach(p => L.marker([p.lat, p.lng]).addTo(map)
                .bindPopup(`<b>${p.order}. ${p.name}</b>${p.city ? '<br>'+p.city : ''}`));

            // --- Routing control (OSRM public) ---
            // Some hosts need explicit serviceUrl. If OSRM fails, we fallback to a straight polyline.
            let routingOk = false;
            const ctrl = L.Routing.control({
                waypoints: wpts,
                router: L.Routing.osrmv1({
                // You can replace by your own OSRM or Mapbox router if needed
                serviceUrl: 'https://router.project-osrm.org/route/v1',
                timeout: 10 * 1000,
                profile: 'driving'
                }),
                show: false,
                addWaypoints: false,
                routeWhileDragging: false,
                draggableWaypoints: false,
                fitSelectedRoutes: true,
            })
            .on('routesfound', () => { routingOk = true; })
            .on('routingerror', () => {
                // fallback: straight polyline between points (not road-matched)
                drawFallbackPolyline();
            })
            .addTo(map);

            // Fallback polyline if routing service is unavailable
            function drawFallbackPolyline() {
                if (!wpts || wpts.length < 2) return;
                const latlngs = wpts.map(ll => [ll.lat, ll.lng]);
                L.polyline(latlngs, { weight: 4, opacity: 0.9 }).addTo(map);
                map.fitBounds(L.latLngBounds(latlngs).pad(0.2));
            }

            // --- Export buttons: Apple Maps & Google Maps ---
            const btnApple = document.getElementById('btn-apple');
            const btnGoogle = document.getElementById('btn-google');

            // Build Apple Maps URL:
            // origin -> daddr chain with "to:" steps, ex:
            // http://maps.apple.com/?saddr=lat0,lng0&daddr=lat1,lng1 to:lat2,lng2 to:latN,lngN
            function buildAppleMapsUrl(points) {
                const origin = points[0];
                const dests  = points.slice(1).map(p => `${p.lat},${p.lng}`);
                const daddr  = dests.length ? dests.join(' to:') : `${origin.lat},${origin.lng}`;
                const url = new URL('http://maps.apple.com/');
                url.searchParams.set('saddr', `${origin.lat},${origin.lng}`);
                url.searchParams.set('daddr', daddr);
                // Optional flags: dirflg=d (driving), w (walking), r (public transit) – not always honored on desktop
                url.searchParams.set('dirflg', 'd');
                return url.toString();
            }

            // Build Google Maps URL:
            // https://www.google.com/maps/dir/?api=1&origin=lat0,lng0&destination=latN,lngN&waypoints=lat1,lng1|lat2,lng2
            function buildGoogleMapsUrl(points) {
                const origin = points[0];
                const dest   = points[points.length - 1];
                const waypoints = points.slice(1, -1).map(p => `${p.lat},${p.lng}`).join('|');
                const url = new URL('https://www.google.com/maps/dir/');
                url.searchParams.set('api', '1');
                url.searchParams.set('travelmode', 'driving');
                url.searchParams.set('origin', `${origin.lat},${origin.lng}`);
                url.searchParams.set('destination', `${dest.lat},${dest.lng}`);
                if (waypoints) url.searchParams.set('waypoints', waypoints);
                return url.toString();
            }

            btnApple?.addEventListener('click', () => {
                if (wpts.length < 1) return;
                // use original items for exact lat/lng
                const url = buildAppleMapsUrl(items);
                window.open(url, '_blank', 'noopener');
            });

            btnGoogle?.addEventListener('click', () => {
                if (wpts.length < 1) return;
                const url = buildGoogleMapsUrl(items);
                window.open(url, '_blank', 'noopener');
            });

            // If OSRM times out (no routesfound), fallback after 12s anyway
            setTimeout(() => { if (!routingOk) drawFallbackPolyline(); }, 12000);
        });
    </script>
@endsection