{{-- resources/views/itins/create.blade.php --}}
@extends('layouts.base')

@section('title', __('Create an itinerary'))

@section('content')
<style>
  .twocol { display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
  .panel { border:var(--border); border-radius:12px; overflow:hidden; }
  .panel-hd { padding:10px 12px; border-bottom:var(--border);  }
  .list { padding:10px; min-height:220px; border-top:var(--border); }
  .row { display:flex; align-items:center; justify-content:space-between; gap:8px; padding:8px 10px; border:1px solid var(--border-color,#1f2430); border-radius:8px; margin-bottom:8px; }
</style>

@php
  $urlAdmins  = route( (request()->routeIs('pro.*') ? 'pro.' : 'admin.').'itins.adminsForSociety', ['locale'=>app()->getLocale()] );
  $urlUsers   = route( (request()->routeIs('pro.*') ? 'pro.' : 'admin.').'itins.usersForSociety', ['locale'=>app()->getLocale()] );
  $urlPoints  = route( (request()->routeIs('pro.*') ? 'pro.' : 'admin.').'itins.pointsForContext', ['locale'=>app()->getLocale()] );
@endphp

<div class="wrap">
  <h2>{{ __('Create an itinerary') }}</h2>

  @if($errors->any())
    <div class="alert danger">
      <ul style="margin:0 0 0 16px;">
        @foreach($errors->all() as $err)<li>{{ $err }}</li>@endforeach
      </ul>
    </div>
  @endif

  <form method="POST" action="{{ route( (request()->routeIs('pro.*') ? 'pro.' : 'admin.').'itins.store', ['locale'=>app()->getLocale()]) }}">
    @csrf

    <div class="card">
      <div class="card-body">
        <div class="form-field">
          <label for="name">{{ __('Name') }} *</label>
          <input id="name" type="text" name="name" required value="{{ old('name') }}" placeholder="Ex: Tournée Sud-Est">
        </div>

        <div class="twocol">
          <div class="form-field">
            <label for="society_id">{{ __('Company') }} *</label>
            <select id="society_id" name="society_id" required>
              <option value="" disabled selected>{{ __('Select…') }}</option>
              @foreach($societies as $s)
                <option value="{{ $s->id }}">{{ $s->name }}</option>
              @endforeach
            </select>
          </div>

          <div class="form-field">
            <label for="assigned_user_id">{{ __('Pro admin') }} *</label>
            <select id="assigned_user_id" name="assigned_user_id" required disabled>
              <option value="">{{ __('Select a company first') }}</option>
            </select>
          </div>
        </div>

        <div class="form-field">
          <label>{{ __('Assigned users (company members)') }}</label>
          <div id="company-users-box" style="display:grid; grid-template-columns: repeat(auto-fit,minmax(240px,1fr)); gap:8px;">
            <div class="muted" data-empty>{{ __('Select a company to list its users.') }}</div>
          </div>
        </div>

        <div class="form-field" style="margin-top:10px;">
          <label>{{ __('Waypoints') }}</label>
          <div class="twocol">
            <div class="panel">
              <div class="panel-hd"><h4>{{ __('Available') }}</h4></div>
              <div id="available-points" class="list">
                <div class="muted" data-empty>{{ __('Select a company and a Pro admin.') }}</div>
              </div>
            </div>
            <div class="panel">
              <div class="panel-hd"><h4>{{ __('Selected (drag to reorder)') }}</h4></div>
              <div id="selected-points" class="list droplist">
                <div class="muted" data-empty>{{ __('No waypoint selected yet.') }}</div>
              </div>
              <div id="point-ids-hidden"></div>
            </div>
          </div>
        </div>

      </div>
      <div class="card-footer actions">
        <a class="btn ghost" href="{{ route( (request()->routeIs('pro.*') ? 'pro.' : 'admin.').'itins.index', ['locale'=>app()->getLocale()]) }}">{{ __('Back') }}</a>
        <button class="btn primary" type="submit">{{ __('Create') }}</button>
      </div>
    </div>
  </form>
</div>

@push('scripts')
    <script>
        document.addEventListener('DOMContentLoaded', () => {
        const rns = {!! json_encode([
            'admins' => $urlAdmins,
            'users'  => $urlUsers,
            'points' => $urlPoints,
        ]) !!};

        const selSoc   = document.getElementById('society_id');
        const selAdm   = document.getElementById('assigned_user_id');
        const usersBox = document.getElementById('company-users-box');
        const availBox = document.getElementById('available-points');
        const selBox   = document.getElementById('selected-points');
        const hidden   = document.getElementById('point-ids-hidden');

        const T = {
            select:               @json(__('Select…')),
            selectCompanyFirst:   @json(__('Select a company first')),
            loading:              @json(__('Loading…')),
            noProForCompany:      @json(__('No pro admin for this company')),
            errAdmins:            @json(__('Error loading admins')),
            noWpForCombo:         @json(__('No waypoint for this company / admin.')),
            errWps:               @json(__('Unable to load waypoints')),
            noUsers:              @json(__('No users found for this company (excluding root).')),
            errUsers:             @json(__('Unable to load users for this company.')),
            noneSelected:         @json(__('No waypoint selected yet.')),
            add:                  @json(__('Add')),
            remove:               @json(__('Remove')),
        };

        // --- tiny helpers ----------------------------------------------------
        function clearEmpty(el){ if(!el) return; const e = el.querySelector('[data-empty]'); if(e) e.remove(); }
        function resetWaypointsUI(){
            if (availBox) availBox.innerHTML = `<div class="muted" data-empty>${T.selectCompanyFirst}</div>`;
            if (selBox)   selBox.innerHTML   = `<div class="muted" data-empty>${T.noneSelected}</div>`;
            if (hidden)   hidden.innerHTML   = '';
        }
        function updateHidden(){
            if(!hidden || !selBox) return;
            hidden.innerHTML = '';
            selBox.querySelectorAll('.row[data-id]').forEach(el => {
            const id = el?.dataset?.id;
            if(!id) return;
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = 'point_ids[]';
            input.value = id; // duplicates allowed
            hidden.appendChild(input);
            });
        }

        // Build rows
        function makeAvailableRow(id, name){
            const div = document.createElement('div');
            div.className   = 'row';
            div.dataset.id  = String(id);
            div.dataset.list= 'available';
            div.draggable   = false;
            div.innerHTML = `
            <span>${name || ('#'+id)}</span>
            <div style="display:flex; gap:6px;">
                <button type="button" class="btn small ghost add">${T.add}</button>
            </div>`;
            return div;
        }
        function makeSelectedRow(id, name){
            const div = document.createElement('div');
            div.className   = 'row';
            div.dataset.id  = String(id);
            div.dataset.list= 'selected';
            div.draggable   = true;
            div.innerHTML = `
            <span>${name || ('#'+id)}</span>
            <div style="display:flex; gap:6px;">
                <button type="button" class="btn small danger remove">${T.remove}</button>
            </div>`;
            return div;
        }

        // Drag only on the right
        (function makeDroppable(container){
            if(!container) return;
            let dragEl = null;
            container.addEventListener('dragstart', (e)=>{
            const row = e.target?.closest?.('.row');
            if(!row || row.dataset.list!=='selected') return;
            dragEl = row;
            e.dataTransfer.effectAllowed = 'move';
            });
            container.addEventListener('dragover', (e)=>{
            if(!dragEl) return;
            e.preventDefault();
            const rows = [...container.querySelectorAll('.row')];
            const after = rows.find(r=>{
                const rect = r.getBoundingClientRect();
                return e.clientY < rect.top + rect.height/2;
            });
            if(after) container.insertBefore(dragEl, after);
            else container.appendChild(dragEl);
            });
            const end = ()=>{ dragEl=null; updateHidden(); };
            container.addEventListener('drop', end);
            container.addEventListener('dragend', end);
        })(selBox);

        // Block drag on the left
        availBox?.addEventListener('dragstart', e => e.preventDefault());

        // Centralized Add/Remove
        document.addEventListener('click', (e)=>{
            const addBtn = e.target?.closest?.('.add');
            if(addBtn){
            e.preventDefault();
            const srcRow = addBtn.closest('.row');
            if(!srcRow) return;
            const id   = srcRow.dataset?.id;
            const name = srcRow.querySelector('span')?.textContent?.trim() || `#${id}`;
            if(!id) return;
            clearEmpty(selBox);
            selBox.appendChild(makeSelectedRow(id, name)); // duplicate on the right (do not remove left)
            updateHidden();
            return;
            }

            const remBtn = e.target?.closest?.('.remove');
            if(remBtn){
            e.preventDefault();
            const row = remBtn.closest('.row');
            if(!row) return;
            row.remove();
            if(!selBox.querySelector('.row')){
                selBox.innerHTML = `<div class="muted" data-empty>${T.noneSelected}</div>`;
            }
            updateHidden();
            }
        });

        // --- load data -------------------------------------------------------
        async function loadAdminsForSociety(societyId){
            selAdm.disabled = true;
            selAdm.innerHTML = `<option value="">${T.loading}</option>`;
            try{
            const res = await fetch(`${rns.admins}?society_id=${encodeURIComponent(societyId)}`, {
                headers:{'Accept':'application/json'}, credentials:'same-origin'
            });
            const txt = await res.text(); let admins=[];
            try{ admins = JSON.parse(txt); }catch{ admins=[]; }

            if(!Array.isArray(admins) || !admins.length){
                selAdm.disabled = true;
                selAdm.innerHTML = `<option value="">${T.noProForCompany}</option>`;
                return;
            }
            selAdm.disabled = false;
            selAdm.innerHTML = `<option value="">${T.select}</option>`;
            admins.forEach(u=>{
                const opt = document.createElement('option');
                const nm  = `${u.firstname||''} ${u.lastname||''}`.trim() || `#${u.id}`;
                opt.value = u.id;
                opt.textContent = nm;
                selAdm.appendChild(opt);
            });
            // auto-select first real admin if you want:
            if(admins[0]?.id){
                selAdm.value = String(admins[0].id);
                selAdm.dispatchEvent(new Event('change'));
            }
            }catch{
            selAdm.disabled = true;
            selAdm.innerHTML = `<option value="">${T.errAdmins}</option>`;
            }
        }

        async function loadUsersForSociety(societyId){
            usersBox.innerHTML = `<div class="muted">${T.loading}</div>`;
            try{
            const res = await fetch(`${rns.users}?society_id=${encodeURIComponent(societyId)}`, {
                headers:{'Accept':'application/json'}, credentials:'same-origin'
            });
            const txt = await res.text(); let users=[];
            try{ users = JSON.parse(txt); }catch{ users=[]; }

            if(!Array.isArray(users) || !users.length){
                usersBox.innerHTML = `<div class="muted">${T.noUsers}</div>`;
                return;
            }
            usersBox.innerHTML = '';
            users.forEach(u=>{
                const lbl = document.createElement('label');
                lbl.className = 'chk';
                lbl.innerHTML = `
                <input type="checkbox" name="assigned_user_ids[]" value="${u.id}">
                <span>${(u.firstname||'')+' '+(u.lastname||'')}</span>
                <small class="muted">— ${u.email||''}</small>
                `;
                usersBox.appendChild(lbl);
            });
            }catch{
            usersBox.innerHTML = `<div class="alert danger">${T.errUsers}</div>`;
            }
        }

        async function loadWaypointsForContext(societyId, adminId){
            if(!availBox) return;
            availBox.innerHTML = `<div class="muted">${T.loading}</div>`;
            try{
            const res = await fetch(`${rns.points}?society_id=${encodeURIComponent(societyId)}&admin_id=${encodeURIComponent(adminId)}`, {
                headers:{'Accept':'application/json'}, credentials:'same-origin'
            });
            const txt = await res.text(); let list=[];
            try{ list = JSON.parse(txt); }catch{ list=[]; }

            if(!Array.isArray(list) || !list.length){
                availBox.innerHTML = `<div class="muted">${T.noWpForCombo}</div>`;
                return;
            }
            availBox.innerHTML = '';
            list.forEach(p=>{
                availBox.appendChild(makeAvailableRow(p.id, p.name || ('#'+p.id)));
            });
            }catch{
            availBox.innerHTML = `<div class="alert danger">${T.errWps}</div>`;
            }
        }

        // --- events ----------------------------------------------------------
        selSoc?.addEventListener('change', async ()=>{
            const sid = selSoc.value;
            resetWaypointsUI();
            if(!sid) {
            selAdm.disabled = true;
            selAdm.innerHTML = `<option value="">${T.selectCompanyFirst}</option>`;
            usersBox.innerHTML = `<div class="muted">${T.selectCompanyFirst}</div>`;
            return;
            }
            await loadAdminsForSociety(sid);
            await loadUsersForSociety(sid);
        });

        selAdm?.addEventListener('change', async ()=>{
            const sid = selSoc?.value;
            const aid = selAdm?.value;
            if(!sid || !aid){
            resetWaypointsUI();
            return;
            }
            await loadWaypointsForContext(sid, aid);
        });

        // Init state
        resetWaypointsUI();
        });
    </script>
@endpush
@endsection