{{-- resources/views/itins/edit.blade.php --}}
@extends('layouts.base')

@section('title', __('Edit itinerary'))

@section('content')
@php
    /** @var \App\Models\Itinerary $itinerary */
    $isPro = request()->routeIs('pro.*');

    $saveUrl  = $isPro
        ? route('pro.itins.update', ['locale'=>app()->getLocale(), 'itinerary'=>$itinerary->id])
        : route('admin.itins.update', ['itinerary'=>$itinerary->id]);

    $indexUrl = $isPro
        ? route('pro.itins.index', ['locale'=>app()->getLocale()])
        : route('admin.itins.index');

    $urlAdmins = $isPro
        ? route('pro.itins.adminsForSociety', ['locale'=>app()->getLocale()])
        : route('admin.itins.adminsForSociety');

    $urlUsers = $isPro
        ? route('pro.itins.usersForSociety', ['locale'=>app()->getLocale()])
        : route('admin.itins.usersForSociety');

    $urlPoints = $isPro
        ? route('pro.itins.pointsForContext', ['locale'=>app()->getLocale()])
        : route('admin.itins.pointsForContext');

    // ==== Chargement sécurisé des sociétés ====
    $authUser = auth()->user();

    if ($isPro) {
        // Un admin pro ne voit que SES sociétés
        // (évite tout SELECT ambigu, et évite null si pas connecté)
        $societies = $societies
            ?? ($authUser ? $authUser->societies()->orderBy('name')->get() : collect());
    } else {
        // L’admin voit toutes les sociétés
        $societies = $societies ?? \App\Models\Society::orderBy('name')->get();
    }

    // ==== Utilisateurs déjà assignés ====
    $assignedUserIds = $assignedUserIds ?? ($itinerary->assignedUsers?->pluck('id')->all() ?? []);

    // ==== Points déjà attachés (tri) ====
    $selectedPoints = $selectedPoints ?? $itinerary->points()
        ->select('points.id', 'points.name')
        ->withPivot(['sort_order'])
        ->orderBy('itinerary_points.sort_order')
        ->get();

    // ==== Contexte initial (seed) ====
    $seedSocietyId = (int) old('society_id', $itinerary->society_id);
    // Si admin pro sans société sur l’itinéraire, on pré-sélectionne sa première société
    if ($isPro && !$seedSocietyId && $societies->count() > 0) {
        $seedSocietyId = (int) $societies->first()->id;
    }
    $seedAdminId   = (int) old('assigned_user_id', $itinerary->assigned_user_id);
@endphp

<style>
    .max-w { max-width: 1200px; margin-inline: auto; }
    .grid-2 { display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
    .panel { border:var(--border); border-radius:12px; overflow:hidden; }
    .panel-hd { padding:10px 12px; border-bottom:var(--border); display:flex; align-items:center; justify-content:space-between;}
    .list { padding:10px; min-height:220px; border-top:var(--border); }
    .row  { display:flex; align-items:center; justify-content:space-between; gap:8px; padding:8px 10px; border:1px solid var(--border-color,#1f2430); border-radius:8px; margin-bottom:8px;}
    .row .btn.small { padding:2px 10px !important; }
    .ul-reset { list-style:none; padding:0; margin:0; }
    .mb-6 { margin-bottom:6px; }
    .muted { color:#a7acb4; font-size:.9em; }
</style>

<div class="wrap">
    <h2 style="margin:0 0 12px;">{{ __('Edit itinerary') }} — <span class="muted">#{{ $itinerary->id }}</span></h2>

    @if(session('ok')) <div class="ok">{{ session('ok') }}</div> @endif

    @if($errors->any())
        <div class="alert danger">
            <ul class="ul-reset">
                @foreach($errors->all() as $err)<li>{{ $err }}</li>@endforeach
            </ul>
        </div>
    @endif

    <form method="POST" action="{{ $saveUrl }}">
        @csrf
        @method('PUT')

        <div class="card max-w">
            <div class="card-body">

                {{-- Nom --}}
                <div class="form-field">
                    <label for="name">{{ __('Name') }} *</label>
                    <input id="name" type="text" name="name" required
                           value="{{ old('name', $itinerary->name) }}"
                           placeholder="Ex: Tournée Sud-Est">
                    <small class="muted">{{ __('A unique itinerary name.') }}</small>
                </div>

                <div class="grid-2">
                    {{-- Société --}}
                    <div class="form-field">
                        <label for="society_id">{{ __('Company') }} *</label>
                        <select id="society_id" name="society_id" required>
                            <option value="">{{ __('Select…') }}</option>
                            @foreach($societies as $s)
                                <option value="{{ $s->id }}" @selected((int)$seedSocietyId === (int)$s->id)>
                                    {{ $s->name }}
                                </option>
                            @endforeach
                        </select>
                        <small class="muted">{{ __('Pick the company this itinerary belongs to.') }}</small>
                    </div>

                    {{-- Admin pro (responsable) --}}
                    <div class="form-field">
                        <label for="assigned_user_id">{{ __('Pro admin') }} *</label>
                        <select id="assigned_user_id" name="assigned_user_id" required {{ $seedSocietyId ? '' : 'disabled' }}>
                            <option value="">{{ $seedSocietyId ? __('Loading…') : __('Select a company first') }}</option>
                        </select>
                        <small class="muted">{{ __('This user will be the itinerary manager.') }}</small>
                    </div>
                </div>

                {{-- Utilisateurs assignés (multi) --}}
                <div class="form-field" style="margin-top:10px;">
                    <label>{{ __('Assigned users (company members)') }}</label>
                    <div id="company-users-box" style="display:grid; grid-template-columns: repeat(auto-fit,minmax(240px,1fr)); gap:8px;">
                        <div class="muted" data-empty>{{ __('Select a company to list its users.') }}</div>
                    </div>
                </div>

                {{-- Waypoints --}}
                <div class="form-field" style="margin-top:10px;">
                    <label>{{ __('Waypoints') }}</label>
                    <div class="muted mb-6">
                        {{ __('Pick a company, then a Pro admin to load waypoints of the company and those shared to that admin. Add and drag to reorder them.') }}
                    </div>

                    <div class="grid-2">
                        {{-- Disponibles --}}
                        <div class="panel">
                            <div class="panel-hd">
                                <h4 style="margin:0">{{ __('Available') }}</h4>
                            </div>
                            <div id="available-points" class="list">
                                <div class="muted" data-empty>{{ $seedSocietyId && $seedAdminId ? __('Loading…') : __('Select a company and a Pro admin.') }}</div>
                            </div>
                        </div>

                        {{-- Sélectionnés --}}
                        <div class="panel">
                            <div class="panel-hd">
                                <h4 style="margin:0">{{ __('Selected (drag to reorder)') }}</h4>
                            </div>
                            <div id="selected-points" class="list droplist">
                                @if($selectedPoints->isEmpty())
                                    <div class="muted" data-empty>{{ __('No waypoint selected yet.') }}</div>
                                @else
                                    @foreach($selectedPoints as $p)
                                        <div class="row" draggable="true" data-id="{{ $p->id }}">
                                            <span>{{ $p->name ?? ('#'.$p->id) }}</span>
                                            <div style="display:flex; gap:6px;">
                                                <button type="button" class="btn small ghost add" style="display:none;">{{ __('Add') }}</button>
                                                <button type="button" class="btn small danger remove">{{ __('Remove') }}</button>
                                            </div>
                                        </div>
                                    @endforeach
                                @endif
                            </div>
                            {{-- on soumet l’ordre via des inputs hidden --}}
                            <div id="point-ids-hidden">
                                @foreach($selectedPoints as $p)
                                    <input type="hidden" name="point_ids[]" value="{{ $p->id }}">
                                @endforeach
                            </div>
                        </div>
                    </div>
                </div>

            </div>

            <div class="card-footer actions">
                <a class="btn ghost" href="{{ $indexUrl }}">{{ __('Back') }}</a>
                <button class="btn primary" type="submit">{{ __('Save') }}</button>
            </div>
        </div>
    </form>
</div>

{{-- Contexte JS minimal et sûr (pas de Blade dans des backticks) --}}
<script>
    window.ITIN_EDIT_CTX = {
        urls: {
            admins: @json($urlAdmins),
            users:  @json($urlUsers),
            points: @json($urlPoints),
        },
        seed: {
            society_id: @json((int)$seedSocietyId),
            admin_id:   @json((int)$seedAdminId),
            selectedUserIds: @json(array_values($assignedUserIds)),
            selectedPoints: @json($selectedPoints->map(fn($p)=>['id'=>$p->id,'name'=>$p->name])->values()),
        },
        i18n: {
            select: @json(__('Select…')),
            selectCompanyFirst: @json(__('Select a company first')),
            loading: @json(__('Loading…')),
            noAdmins: @json(__('No pro admin for this company')),
            noUsers: @json(__('No users found for this company (excluding root).')),
            add: @json(__('Add')),
            remove: @json(__('Remove')),
            noneSelected: @json(__('No waypoint selected yet.')),
            selectCompanyAndAdmin: @json(__('Select a company and a Pro admin.')),
            unableLoadPoints: @json(__('Unable to load waypoints')),
            unableLoadUsers:  @json(__('Unable to load users for this company.')),
        }
    };
</script>

<script>
    document.addEventListener('DOMContentLoaded', () => {
        const C = window.ITIN_EDIT_CTX;

        const selectSoc   = document.getElementById('society_id');
        const selectAdmin = document.getElementById('assigned_user_id');
        const usersBox    = document.getElementById('company-users-box');

        const availBox = document.getElementById('available-points');
        const selBox   = document.getElementById('selected-points');
        const hidden   = document.getElementById('point-ids-hidden');

        // Helpers
        const clearEmpty = (el) => { const e = el.querySelector('[data-empty]'); if (e) e.remove(); };
        const makeRow = (id, name, inSelected=false) => {
            const div = document.createElement('div');
            div.className = 'row';
            div.draggable = true;
            div.dataset.id = String(id);
            div.innerHTML = `
                <span>${name || ('#'+id)}</span>
                <div style="display:flex; gap:6px;">
                    <button type="button" class="btn small ghost add" ${inSelected ? 'style="display:none;"' : ''}>${C.i18n.add}</button>
                    <button type="button" class="btn small danger remove" ${inSelected ? '' : 'style="display:none;"'}>${C.i18n.remove}</button>
                </div>`;
            return div;
        };
        const updateHidden = () => {
            hidden.innerHTML = '';
            selBox.querySelectorAll('.row[data-id]').forEach((el) => {
                const id = el.dataset.id;
                const input = document.createElement('input');
                input.type = 'hidden';
                input.name = 'point_ids[]';
                input.value = id;
                hidden.appendChild(input);
            });
        };
        const makeDroppable = (container) => {
            let dragEl = null;
            container.addEventListener('dragstart', (e) => {
                const row = e.target.closest('.row');
                if (!row) return;
                dragEl = row;
                e.dataTransfer.effectAllowed = 'move';
            });
            container.addEventListener('dragover', (e) => {
                if (!dragEl) return;
                e.preventDefault();
                const after = [...container.querySelectorAll('.row')].find(r => {
                    const rect = r.getBoundingClientRect();
                    return e.clientY < rect.top + rect.height/2;
                });
                if (after) container.insertBefore(dragEl, after); else container.appendChild(dragEl);
            });
            container.addEventListener('drop', () => { dragEl = null; updateHidden(); });
            container.addEventListener('dragend', () => { dragEl = null; updateHidden(); });
        };
        makeDroppable(selBox);

        // Click handlers Add/Remove
        document.body.addEventListener('click', (e) => {
            const addBtn = e.target.closest('.add');
            const remBtn = e.target.closest('.remove');
            if (addBtn) {
                const row = addBtn.closest('.row');
                if (!row) return;
                clearEmpty(selBox);
                row.querySelector('.add').style.display = 'none';
                row.querySelector('.remove').style.display = '';
                selBox.appendChild(row);
                updateHidden();
            }
            if (remBtn) {
                const row = remBtn.closest('.row');
                if (!row) return;

                row.querySelector('.add').style.display = '';
                row.querySelector('.remove').style.display = 'none';

                clearEmpty(availBox);
                availBox.appendChild(row);

                if (!selBox.querySelector('.row')) {
                    selBox.innerHTML = `<div class="muted" data-empty>${C.i18n.noneSelected}</div>`;
                }
                updateHidden();
            }
        });

        // Load admins for society
        async function loadAdmins(societyId, preselectId = null) {
            selectAdmin.disabled = true;
            selectAdmin.innerHTML = `<option value="">${C.i18n.loading}</option>`;
            try {
                const res = await fetch(`${C.urls.admins}?society_id=${encodeURIComponent(societyId)}`, {
                    headers: { 'Accept':'application/json', 'X-Requested-With':'XMLHttpRequest' },
                    credentials: 'same-origin',
                });
                const text = await res.text(); let admins = [];
                try { admins = JSON.parse(text); } catch { admins = []; }
                if (!Array.isArray(admins) || admins.length === 0) {
                    selectAdmin.innerHTML = `<option value="">${C.i18n.noAdmins}</option>`;
                    return;
                }
                selectAdmin.innerHTML = `<option value="">${C.i18n.select}</option>`;
                admins.forEach(u => {
                    const opt = document.createElement('option');
                    opt.value = u.id;
                    opt.textContent = [u.firstname, u.lastname].filter(Boolean).join(' ') || `#${u.id}`;
                    if (preselectId && Number(preselectId) === Number(u.id)) opt.selected = true;
                    selectAdmin.appendChild(opt);
                });
                selectAdmin.disabled = false;
            } catch (err) {
                console.error('adminsForSociety', err);
                selectAdmin.innerHTML = `<option value="">${C.i18n.selectCompanyFirst}</option>`;
            }
        }

        // Load users for society
        async function loadUsers(societyId, preselectedIds = []) {
            usersBox.innerHTML = `<div class="muted">${C.i18n.loading}</div>`;
            try {
                const res = await fetch(`${C.urls.users}?society_id=${encodeURIComponent(societyId)}`, {
                    headers: { 'Accept':'application/json', 'X-Requested-With':'XMLHttpRequest' },
                    credentials: 'same-origin',
                });
                const text = await res.text(); let users = [];
                try { users = JSON.parse(text); } catch { users = []; }
                if (!Array.isArray(users) || users.length === 0) {
                    usersBox.innerHTML = `<div class="muted">${C.i18n.noUsers}</div>`;
                    return;
                }
                usersBox.innerHTML = '';
                users.forEach(u => {
                    const id = u.id;
                    const chk = document.createElement('label');
                    chk.className = 'chk';
                    const name = [u.firstname, u.lastname].filter(Boolean).join(' ') || `#${id}`;
                    const checked = preselectedIds.includes(id) || preselectedIds.includes(Number(id));
                    chk.innerHTML = `
                        <input type="checkbox" name="assigned_user_ids[]" value="${id}" ${checked ? 'checked' : ''}>
                        <span>${name}</span> <small class="muted">— ${u.email || ''}</small>
                    `;
                    usersBox.appendChild(chk);
                });
            } catch (err) {
                console.error('usersForSociety', err);
                usersBox.innerHTML = `<div class="alert danger">${C.i18n.unableLoadUsers}</div>`;
            }
        }

        // Load points for (society + admin)
        async function loadPoints(societyId, adminId) {
            const availBox = document.getElementById('available-points');
            availBox.innerHTML = `<div class="muted">${C.i18n.loading}</div>`;
            try {
                const url = `${C.urls.points}?society_id=${encodeURIComponent(societyId)}&admin_id=${encodeURIComponent(adminId)}&itinerary_id={{ $itinerary->id }}`;
                const res = await fetch(url, {
                    headers: { 'Accept':'application/json', 'X-Requested-With':'XMLHttpRequest' },
                    credentials: 'same-origin',
                });
                const text = await res.text(); let list = [];
                try { list = JSON.parse(text); } catch { list = []; }

                // Filtrer ceux déjà dans la colonne de droite
                const selBox = document.getElementById('selected-points');
                const selectedIds = new Set([...selBox.querySelectorAll('.row[data-id]')].map(x => Number(x.dataset.id)));

                availBox.innerHTML = '';
                if (!Array.isArray(list) || list.length === 0) {
                    availBox.innerHTML = `<div class="muted">${C.i18n.selectCompanyAndAdmin}</div>`;
                    return;
                }
                list.forEach(p => {
                    if (selectedIds.has(Number(p.id))) return;
                    availBox.appendChild(makeRow(p.id, p.name || ('#'+p.id), false));
                });
            } catch (err) {
                console.error('pointsForContext', err);
                availBox.innerHTML = `<div class="alert danger">${C.i18n.unableLoadPoints}</div>`;
            }
        }

        // Seed selected points (déjà fait en serveur), rien à faire ici.

        // Seed : si société existe, charge admins + users
        if (C.seed.society_id) {
            loadAdmins(C.seed.society_id, C.seed.admin_id);
            loadUsers(C.seed.society_id, C.seed.selectedUserIds || []);
        }

        // Seed : si société & admin existent, charge waypoints disponibles
        if (C.seed.society_id && C.seed.admin_id) {
            loadPoints(C.seed.society_id, C.seed.admin_id);
        }

        // Changement société → recharge admins, users, reset points
        selectSoc?.addEventListener('change', () => {
            const sid = selectSoc.value;
            // reset admin select
            selectAdmin.disabled = true;
            selectAdmin.innerHTML = `<option value="">${C.i18n.loading}</option>`;
            // reset listes
            availBox.innerHTML = `<div class="muted" data-empty>${C.i18n.selectCompanyAndAdmin}</div>`;
            // users
            loadAdmins(sid, null);
            loadUsers(sid, []);
            // on ne charge pas les points tant que l’admin pro n’est pas choisi
        });

        // Changement admin pro → charge points
        selectAdmin?.addEventListener('change', () => {
            const sid = selectSoc.value;
            const aid = selectAdmin.value;
            if (!sid || !aid) return;
            loadPoints(sid, aid);
        });
    });
</script>
@endsection