@extends('layouts.base')

@section('title','Mes waypoints')

@section('content')

    {{-- Leaflet CSS (CDN) --}}
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css">
    <link rel="stylesheet" href="https://unpkg.com/leaflet.locatecontrol@0.84.2/dist/L.Control.Locate.min.css">

  
    <div class="map-page">
        <div class="map-hero">
            <h1>{{ __('Create a waypoint') }}</h1>
            <p>{{ __('Click on the map to pre-fill the form, then complete the waypoint details and save.') }}</p>
        </div>

        @if(session('ok'))
            <div class="alert success" style="margin-bottom: 18px">{{ session('ok') }}</div>
        @endif

        <div id="map"></div>
        <div id="locationStatus" class="location-status" role="status" aria-live="polite"></div>

        <form id="point-form" class="form-card" method="POST" action="{{ lr('points.store') }}" enctype="multipart/form-data">
            @csrf
            <input type="hidden" name="_method" id="methodSpoof" value="POST">
            <input type="hidden" id="currentId" value="">
            <input type="hidden" name="favoris" value="0">

            <div class="form-field">
                <label for="name">{{ __('Name') }} *</label>
                <input type="text" name="name" id="name" required placeholder="Ex. Point A" value="{{ old('name') }}">
                <p class="form-hint">{{ __('Each waypoint name must be unique.') }}</p>
                <p id="duplicateMessage" class="duplicate-message">{{ __('This name is already used. Choose another name.') }}</p>
            </div>

            <div class="form-grid">
                <div class="form-field">
                    <label for="lat">{{ __('Latitude') }} *</label>
                    <input id="lat" type="number" step="0.0000001" name="latitude" required value="{{ old('latitude') }}">
                </div>
                <div class="form-field">
                    <label for="lng">{{ __('Longitude') }} *</label>
                    <input id="lng" type="number" step="0.0000001" name="longitude" required value="{{ old('longitude') }}">
                </div>
            </div>

            <div class="form-grid">
                <div class="form-field">
                    <label for="alt">{{ __('Altitude') }} (m)</label>
                    <input id="alt" type="number" step="0.01" name="altitude" value="{{ old('altitude') }}">
                </div>
                <div class="form-field">
                    <label for="floor">{{ __('Floor') }}</label>
                    <input id="floor" type="number" name="floor" value="{{ old('floor', 0) }}">
                </div>
            </div>

            <div class="form-grid">
                <div class="form-field">
                    <label for="address1">{{ __('Address 1') }}</label>
                    <input id="address1" type="text" name="address1" value="{{ old('address1') }}">
                </div>
                <div class="form-field">
                    <label for="address2">{{ __('Address 2') }}</label>
                    <input id="address2" type="text" name="address2" value="{{ old('address2') }}">
                </div>
            </div>

            <div class="form-grid form-grid--3">
                <div class="form-field">
                    <label for="postcode">{{ __('Postcode') }}</label>
                    <input id="postcode" type="text" name="postcode" value="{{ old('postcode') }}">
                </div>
                <div class="form-field">
                    <label for="city">{{ __('City') }}</label>
                    <input id="city" type="text" name="city" value="{{ old('city') }}">
                </div>
                <div class="form-field">
                    <label for="country_isocode">{{ __('country_isocode') }}</label> 
                    <input id="country_isocode" type="text" name="country_isocode" maxlength="3" value="{{ old('country_isocode') }}" placeholder="Ex. FR, US, DE">
                </div>
            </div>
 
            <div class="form-field">
                <div class="form-field">
                    <label for="comment">{{ __('Comment') }}</label>
                    <input id="comment" type="text" name="comment" value="{{ old('comment') }}">
                </div>
               <label for="photo">{{ __('Photo') }} ({{ __('optional') }})</label>
                <input type="file" id="photo" name="photo" accept="image/*">
            </div>

            <div class="actions">
                <button type="submit" id="saveBtn" class="btn primary">{{ __('Save') }}</button>
                <button type="button" id="cancelBtn" class="btn ghost" style="display:none">{{ __('Cancel') }}</button>
                <button type="button" id="deleteBtn" class="btn danger" style="display:none">{{ __('Delete') }}</button>
            </div>
        </form>

        <form id="delete-form" method="POST" style="display:none">
            @csrf @method('DELETE')
        </form>
    </div>

    {{-- Leaflet JS (CDN) – load before your script --}}
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <script src="https://unpkg.com/leaflet.locatecontrol@0.84.2/dist/L.Control.Locate.min.js"></script>

    <script>
    (function(){
      // Normalise $points quand c’est une pagination
      const _rawPoints = @json($points ?? []);
      const points = Array.isArray(_rawPoints?.data) ? _rawPoints.data
                    : (Array.isArray(_rawPoints) ? _rawPoints : []);
      const existingNames = points
        .filter(p => typeof p?.name === 'string' && p.name.trim().length)
        .map(p => ({ id: String(p.id ?? ''), name: p.name.trim().toLowerCase() }));

      const duplicateMessage = document.getElementById('duplicateMessage');
      const currentIdField = document.getElementById('currentId');
      const locationStatus = document.getElementById('locationStatus');

      function showLocationMessage(message) {
        if (!locationStatus) return;
        if (!message) {
          locationStatus.textContent = '';
          locationStatus.classList.remove('is-visible');
          return;
        }

        locationStatus.textContent = message;
        locationStatus.classList.add('is-visible');
      }

        // ⚙️ enforce icon URLs and size/anchors when using local assets (no CDN)

        // 1) Local folder (ensure public/assets/leaflet/{marker-icon.png, marker-icon-2x.png, marker-shadow.png} exists)
        const ICON_BASE = `${location.origin}/assets/leaflet/`;

        // 2) Disable Leaflet automatic path resolution (otherwise it prefixes with .../dist/images/)
        delete L.Icon.Default.prototype._getIconUrl;

        // 3) Force absolute URLs + explicit sizes
        L.Icon.Default.mergeOptions({
            iconRetinaUrl: ICON_BASE + 'marker-icon-2x.png',
            iconUrl:       ICON_BASE + 'marker-icon.png',
            shadowUrl:     ICON_BASE + 'marker-shadow.png',
            iconSize:      [25, 41],
            iconAnchor:    [12, 41],
            popupAnchor:   [1, -34],
            shadowSize:    [41, 41],
        });

  
     
      // Initialize the map
      const map = L.map('map').setView([46.5, 2.5], 6);
      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; OpenStreetMap contributors', maxZoom: 19
      }).addTo(map);

      // Locate control (single instance)
      L.control.locate({
        position: 'topleft',
        flyTo: true,
        strings: {
          title: '{{ __('Center on my position') }}'
        }
      }).addTo(map);

      map.on('locationerror', () => {
        showLocationMessage('{{ __('Geolocation failed: please allow location access or fill the form manually.') }}');
      });

      map.on('locationfound', () => {
        showLocationMessage(null);
      });

      // Existing markers
      const bounds = L.latLngBounds([]);
      points.forEach(p => {
        const lat = Number(p.latitude), lng = Number(p.longitude);
        if (Number.isFinite(lat) && Number.isFinite(lng)) {
          const m = L.marker([lat, lng]).addTo(map);
          let html = `<b>${p.name ?? '—'}</b>`;
          if (p.address1) html += `<br>${p.address1}`;
          if (p.postcode || p.city) html += `<br>${p.postcode ?? ''} ${p.city ?? ''}`;
          m.bindPopup(html);
          bounds.extend([lat, lng]);
        }
      });
      if (points.length && bounds.isValid()) map.fitBounds(bounds.pad(0.2));

      // Form elements
      const f = {
        form:      document.getElementById('point-form'),
        name:      document.getElementById('name'),
        lat:       document.getElementById('lat'),
        lng:       document.getElementById('lng'),
        alt:       document.getElementById('alt'),
        address1:  document.getElementById('address1'),
        address2:  document.getElementById('address2'),
        postcode:  document.getElementById('postcode'),
        city:      document.getElementById('city'),
        country:   document.getElementById('country_isocode'),
      };

      function checkNameUniqueness() {
        if (!f.name) return;
        const currentName = (f.name.value || '').trim().toLowerCase();
        const currentId = (currentIdField?.value || '').trim();

        if (!currentName) {
          f.name.setCustomValidity('');
          duplicateMessage?.classList.remove('is-visible');
          return;
        }

        const duplicate = existingNames.some(entry => {
          if (!entry.name) return false;
          if (currentId && entry.id === currentId) {
            return false;
          }
          return entry.name === currentName;
        });

        if (duplicate) {
          f.name.setCustomValidity('{{ __('This name is already used. Choose another name.') }}');
          duplicateMessage?.classList.add('is-visible');
        } else {
          f.name.setCustomValidity('');
          duplicateMessage?.classList.remove('is-visible');
        }
      }

      if (f.name) {
        f.name.addEventListener('input', checkNameUniqueness);
        f.name.addEventListener('blur', checkNameUniqueness);
      }

      if (f.form) {
        f.form.addEventListener('submit', (event) => {
          checkNameUniqueness();
          if (!f.name.checkValidity()) {
            event.preventDefault();
            f.name.reportValidity();
          }
        });
      }

      // Reset the form to “create” mode
      function resetFormForCreate() {
        if (!f.form) return;
        f.form.action = "{{ lr('points.store', ['id' => '']) }}";
        const methodSpoof = document.getElementById('methodSpoof');
        if (methodSpoof) methodSpoof.value = 'POST';
        if (currentIdField) currentIdField.value = '';
        if (f.name) {
          f.name.value = '';
          checkNameUniqueness();
        }
        if (f.address1) f.address1.value = '';
        if (f.address2) f.address2.value = '';
        if (f.postcode) f.postcode.value = '';
        if (f.city)     f.city.value = '';
        if (f.country)  f.country.value = '';
        const comment = document.getElementById('comment');
        if (comment) comment.value = '';
      }

      // Temporary marker
      let clickMarker = null;

      // MAP CLICK → populate the form
      map.on('click', async (ev) => {
        const lat = ev?.latlng?.lat;
        const lng = ev?.latlng?.lng;
        if (!Number.isFinite(lat) || !Number.isFinite(lng)) return;

        resetFormForCreate();
        showLocationMessage(null);
        if (f.lat) f.lat.value = lat.toFixed(7);
        if (f.lng) f.lng.value = lng.toFixed(7);
        if (f.alt) f.alt.value = '';

        // Altitude
        let elevation = null;
        try { elevation = await getElevationMeters(lat, lng); }
        catch (err) { console.debug('elevation fail', err); }
        if (elevation !== null && f.alt) f.alt.value = Number(elevation).toFixed(2);

        // Address
        try {
          const addr = await reverseGeocodeFull(lat, lng);
          if (addr) {
            if (f.address1) f.address1.value = addr.address1 || '';
            if (f.address2) f.address2.value = addr.address2 || '';
            if (f.postcode) f.postcode.value = addr.postcode || '';
            if (f.city)     f.city.value     = addr.city || '';
            if (f.country)  f.country.value  = addr.country_isocode || '';
          }
        } catch (err) { console.debug('reverse geocode fail', err); }

        // Visual marker
        if (clickMarker) map.removeLayer(clickMarker);
        const popupHtml = `
          <b>Nouveau point</b><br>
          Lat: ${lat.toFixed(5)}<br>
          Lng: ${lng.toFixed(5)}<br>
          Alt: ${elevation !== null ? Number(elevation).toFixed(2)+' m' : '—'}
        `;
        clickMarker = L.marker([lat, lng], { opacity: 0.95 }).addTo(map).bindPopup(popupHtml).openPopup();
      });

      // Altitude & reverse geocode helpers
      async function fetchElevationOpenElevation(lat, lng) {
        const r = await fetch(`https://api.open-elevation.com/api/v1/lookup?locations=${lat},${lng}`).catch(()=>null);
        if (!r || !r.ok) throw new Error('OpenElevation error');
        const j = await r.json();
        const el = j?.results?.[0]?.elevation;
        if (typeof el !== 'number') throw new Error('No elevation');
        return el;
      }
      async function fetchElevationOpenTopo(lat, lng) {
        const r = await fetch(`https://api.opentopodata.org/v1/srtm90m?locations=${lat},${lng}`).catch(()=>null);
        if (!r || !r.ok) throw new Error('OpenTopo error');
        const j = await r.json();
        const el = j?.results?.[0]?.elevation;
        if (typeof el !== 'number') throw new Error('No elevation');
        return el;
      }
      async function getElevationMeters(lat, lng) {
        try { return await fetchElevationOpenElevation(lat,lng); }
        catch (_) { try { return await fetchElevationOpenTopo(lat,lng); } catch (_) { return null; } }
      }
      async function reverseGeocodeFull(lat, lng) {
        const r = await fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lng}&zoom=18&addressdetails=1`,
          { headers:{ 'Accept':'application/json' } }).catch(()=>null);
        if (!r || !r.ok) return null;
        const j = await r.json();
        const a = j.address || {};
        const address1 = `${a.house_number ? a.house_number+' ' : ''}${a.road || ''}`.trim();
        const address2 = a.suburb || a.neighbourhood || '';
        const city     = a.city || a.town || a.village || a.hamlet || '';
        const postcode = a.postcode || '';
        const cc       = (a.country_code || '').toUpperCase();
        return { address1, address2, postcode, city, country_isocode: cc };
      }
    })();
  </script>

@endsection
