{{-- resources/views/partials/lang-switcher.blade.php --}}
@php
    $allLocales = config('app.locales', []);
    $currentLocale = app()->getLocale();

    // On récupère la route courante + ses paramètres
    $currentRouteName = \Illuminate\Support\Facades\Route::currentRouteName();
    $currentParams    = \Illuminate\Support\Facades\Route::current()?->parameters() ?? [];

    // La locale est-elle un paramètre attendu par la route courante ?
    $routeExpectsLocale = array_key_exists('locale', $currentParams);

    // Helper pour fabriquer l'URL localisée de la page courante.
    // Priorités :
    // - si la route a un paramètre {locale} -> on remplace juste
    // - sinon, si tu as un helper lr() dispo -> on l’essaie
    // - sinon, fallback basique: /{locale}
    $urlFor = function (string $code) use ($currentRouteName, $currentParams, $routeExpectsLocale) {
        try {
            if ($currentRouteName) {
                $params = $currentParams;
                if ($routeExpectsLocale) {
                    $params['locale'] = $code;
                    return route($currentRouteName, $params);
                }
                // Si ta base utilise un helper lr() pour les routes localisées, on essaie
                if (function_exists('lr')) {
                    return lr($currentRouteName, $params + ['locale' => $code]);
                }
                // Route nommée sans {locale} → tente quand même
                return route($currentRouteName, $params);
            }
        } catch (\Throwable $e) {
            // on tombe sur le fallback
        }
        // Fallback minimal : racine locale
        return url('/'.$code);
    };
@endphp

<div class="lang-switcher">
    <button class="lang-btn" type="button" aria-haspopup="listbox" aria-expanded="false">
        <span class="flag">{{ $allLocales[$currentLocale]['flag'] ?? '🌐' }}</span>
        <span class="label">{{ $allLocales[$currentLocale]['name'] ?? strtoupper($currentLocale) }}</span>
        <svg class="chevron" width="14" height="14" viewBox="0 0 24 24" aria-hidden="true"><path d="M6 9l6 6 6-6"/></svg>
    </button>

    <ul class="lang-menu" role="listbox" tabindex="-1">
        @foreach($allLocales as $code => $meta)
            <li role="option" aria-selected="{{ $code === $currentLocale ? 'true' : 'false' }}">
                <a href="{{ $urlFor($code) }}" class="lang-item {{ $code === $currentLocale ? 'active' : '' }}">
                    <span class="flag">{{ $meta['flag'] ?? '🌐' }}</span>
                    <span class="name">{{ $meta['name'] ?? strtoupper($code) }}</span>
                </a>
            </li>
        @endforeach
    </ul>
</div>

<script>
    document.addEventListener('DOMContentLoaded', () => {
        // Gère tous les switchers présents sur la page
        document.querySelectorAll('.lang-switcher').forEach((box) => {
            const btn  = box.querySelector('.lang-btn');
            const menu = box.querySelector('.lang-menu');
            if (!btn || !menu) return;

            const open  = () => { menu.classList.add('open'); btn.setAttribute('aria-expanded', 'true'); };
            const close = () => { menu.classList.remove('open'); btn.setAttribute('aria-expanded', 'false'); };
            const toggle= () => { menu.classList.contains('open') ? close() : open(); };

            // Ouverture / fermeture au clic bouton
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                toggle();
                if (menu.classList.contains('open')) {
                    // focus sur le premier lien pour l’accessibilité
                    const first = menu.querySelector('a, button, [tabindex]:not([tabindex="-1"])');
                    first && first.focus();
                }
            });

            // Clic en dehors -> ferme
            document.addEventListener('click', (e) => {
                if (!box.contains(e.target)) close();
            });

            // Échappe -> ferme
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape') close();
            });

            // Navigation clavier bas/haut dans la liste
            menu.addEventListener('keydown', (e) => {
                const items = Array.from(menu.querySelectorAll('a.lang-item'));
                const idx   = items.indexOf(document.activeElement);
                if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    (items[idx + 1] || items[0])?.focus();
                } else if (e.key === 'ArrowUp') {
                    e.preventDefault();
                    (items[idx - 1] || items[items.length - 1])?.focus();
                }
            });
        });
    });
</script>