<?php

namespace Tests\Unit\Controllers;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;
use Tests\TestCase;

class ControllerCoverageTest extends TestCase
{
    /**
     * Ensure every controller class under App\Http\Controllers resolves via the container.
     */
    public function test_controllers_resolve_through_service_container(): void
    {
        foreach ($this->discoverControllers() as $class) {
            $instance = app()->make($class);
            $this->assertInstanceOf(
                $class,
                $instance,
                "Unable to resolve controller {$class} from the service container."
            );
        }
    }

    /**
     * Ensure every route referencing a controller points to an existing class + method.
     */
    public function test_routes_reference_existing_controller_methods(): void
    {
        foreach (Route::getRoutes() as $route) {
            $controllerAction = $route->getAction()['controller'] ?? null;

            if (!$controllerAction || $controllerAction instanceof \Closure) {
                continue;
            }

            [$class, $method] = $this->splitControllerAction($controllerAction);

            if (!$class) {
                continue;
            }

            $this->assertTrue(
                class_exists($class),
                sprintf('Route [%s] references missing controller [%s].', $route->uri(), $class)
            );

            $this->assertTrue(
                method_exists($class, $method),
                sprintf('Route [%s] references missing method [%s::%s].', $route->uri(), $class, $method)
            );
        }
    }

    /**
     * Discover fully-qualified controller class names.
     *
     * @return array<int, string>
     */
    private function discoverControllers(): array
    {
        $basePath = app_path('Http/Controllers');

        return collect(File::allFiles($basePath))
            ->filter(fn($file) => Str::endsWith($file->getFilename(), 'Controller.php'))
            ->map(function ($file) use ($basePath) {
                $relative = Str::after($file->getPathname(), $basePath . DIRECTORY_SEPARATOR);

                return 'App\\Http\\Controllers\\' . str_replace(
                    [DIRECTORY_SEPARATOR, '.php'],
                    ['\\', ''],
                    $relative
                );
            })
            ->unique()
            ->values()
            ->all();
    }

    /**
     * Split a Laravel controller action string/array into class + method.
     *
     * @param mixed $controllerAction
     * @return array{0: string|null, 1: string}
     */
    private function splitControllerAction(mixed $controllerAction): array
    {
        if (is_string($controllerAction)) {
            if (Str::contains($controllerAction, '@')) {
                [$class, $method] = explode('@', $controllerAction, 2);

                return [$class, $method];
            }

            return [$controllerAction, '__invoke'];
        }

        if (is_array($controllerAction) && isset($controllerAction[0])) {
            $class = is_string($controllerAction[0]) ? $controllerAction[0] : null;
            $method = $controllerAction[1] ?? '__invoke';

            return [$class, $method];
        }

        return [null, '__invoke'];
    }
}

